<?php
namespace App\Http\Models\Api;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class StockSellStorageGoods extends Model
{
    /**
     * [purchaseSales 商品采购销出，进销存成本计算]
     * @param  [type]  $user_info      [用户信息]
     * @param  [type]  $algorithm_type [进销存计算方式：1先进先出2移动加权平均法3无成本]
     * @param  [type]  $type           [类型：1采购，2销出，3调拨出库，4调拨入库5采购退货出库]
     * @param  [type]  $ware_id        [仓库id]
     * @param  [type]  $waon_class     [仓库类别]
     * @param  [type]  $goo_id         [数组，商品id]
     * @param  [type]  $goo_quantity   [数组，商品数量；[商品id=>商品数量]]
     * @param  [type]  $goo_price      [数组，商品金额，成本价(不乘以100)；[商品id=>商品金额]]
     * @param  integer $average_type   [进销存的移动加权平均法算法类型：1计算平均价2不计算平均价【退换回收时不计算平均价】]
     * @return [type]                  [description]
     */
    public static function purchaseSales(
        $user_info,
        $algorithm_type,
        $type,
        $ware_id,
        $waon_class,
        $goo_id,
        $goo_quantity,
        $goo_price,
        $average_type=1
    ) {
        //门店id
        $shop_id = $user_info->shop_superior_id ? $user_info->shop_superior_id : $user_info->adm_shop_id;
        $now_year = date('Y');//当前年
        $now_month = date('m');//当前月

        $last_month = $now_month - 1;//上个月
        //如果当前月是一月份，上个月的数据就为0
        if ($last_month == 0) {
            // 上个月就是去年的12月
            $last_month = 12;
            $last_year = $now_year - 1;
        } else {
            $last_year = $now_year;
        }
        
        // 进销存算法，先进先出
        if ($algorithm_type == 1) {
            return self::firstInFirstOut(
                $user_info,
                $shop_id,
                $type,
                $ware_id,
                $waon_class,
                $goo_id,
                $goo_quantity,
                $goo_price,
                $now_year,
                $now_month,
                $last_year,
                $last_month
            );
        } elseif ($algorithm_type == 2) {
            // 移动加权平均法
            if ($average_type == 1) {
                return self::movingWeightedAverage(
                    $user_info,
                    $shop_id,
                    $type,
                    $ware_id,
                    $waon_class,
                    $goo_id,
                    $goo_quantity,
                    $goo_price,
                    $now_year,
                    $now_month,
                    $last_year,
                    $last_month
                );
            // 退换回收，移动加权平均法计算库存，但不计算平均价
            } else {
                return self::movingWeightedNoAverage(
                    $user_info,
                    $shop_id,
                    $type,
                    $ware_id,
                    $waon_class,
                    $goo_id,
                    $goo_quantity,
                    $goo_price,
                    $now_year,
                    $now_month,
                    $last_year,
                    $last_month
                );
            }
        } elseif ($algorithm_type == 3) {
            // 无成本
            return self::noCost(
                $user_info,
                $shop_id,
                $type,
                $ware_id,
                $waon_class,
                $goo_id,
                $goo_quantity,
                $goo_price,
                $now_year,
                $now_month,
                $last_year,
                $last_month
            );
        }
    }

    /**
     * [firstInFirstOut 进销存先进先出算法]
     * @param  [type] $user_info    [用户信息]
     * @param  [type] $shop_id      [总店id]
     * @param  [type] $type         [类型：1采购，2销出，3调拨出库，4调拨入库]
     * @param  [type] $ware_id      [仓库id]
     * @param  [type] $waon_class   [仓库类别]
     * @param  [type] $goo_id       [数组，商品id]
     * @param  [type] $goo_quantity [数组，商品数量；[商品id=>商品数量]]
     * @param  [type] $goo_price    [数组，商品金额，成本价(不乘以100)；[商品id=>商品金额]]
     * @param  [type] $now_year     [当前年份]
     * @param  [type] $now_month    [当前月]
     * @param  [type] $last_year    [上个月的年份]
     * @param  [type] $last_month   [上个月]
     * @return [type]               [description]
     */
    protected static function firstInFirstOut(
        $user_info,
        $shop_id,
        $type,
        $ware_id,
        $waon_class,
        $goo_id,
        $goo_quantity,
        $goo_price,
        $now_year,
        $now_month,
        $last_year,
        $last_month
    ) {
        $goo_all_amount = 0;//商品总数量
        $goo_all_price = 0;//商品总金额
        // 先进先出，进销存商品记录表，记录商品数据
        foreach ($goo_id as $key => $value) {
            $now_cost = $goo_quantity[$value] * $goo_price[$value] * 100;
            $goo_all_amount += $goo_quantity[$value];
            if ($type == 1) {
                $goo_all_price += $now_cost;
                // 采购
                // 先进先出，商品记录数据
                $ssgr_data['ssgr_cid'] = $user_info->adm_cid;//公司id
                $ssgr_data['ssgr_superior_id'] = $shop_id;//所属总店id
                $ssgr_data['ssgr_shop_id'] = $user_info->adm_shop_id;//	所属门店id
                $ssgr_data['ssgr_ware_id'] = $ware_id;//仓库id
                $ssgr_data['ssgr_waon_class'] = $waon_class;//仓库类别
                $ssgr_data['ssgr_goo_id'] = $value;//商品id
                $ssgr_data['ssgr_year'] = $now_year;//年份
                $ssgr_data['ssgr_month'] = $now_month;//月
                $ssgr_data['ssgr_initial_stock'] = $goo_quantity[$value];//期初库存，总库存数量
                $ssgr_data['ssgr_initial_average_cost'] = $goo_price[$value] * 100;//期初平均价（单位：分）
                $ssgr_data['ssgr_initial_cost'] = $now_cost;//期初金额-成本（单位：分）,总的成本
                $ssgr_data['ssgr_surplus_stock'] = $goo_quantity[$value];//结余库存，剩余总库存数量
                $ssgr_data['ssgr_surplus_average_cost'] = $goo_price[$value] * 100;//期末平均价（单位：分）
                $ssgr_data['ssgr_surplus_cost'] = $now_cost;//结余成本（单位：分），剩余总的成本
                $ssgr_data['ssgr_month_enter_stock'] = $goo_quantity[$value];//月入库存数量
                $ssgr_data['ssgr_month_enter_cost'] = $now_cost;//月入成本（单位：分）
                $ssgr_data['ssgr_update_time'] = date('Y-m-d H:i:s');
                $ssgr_res = DB::table('stock_sell_goods_record')->insert($ssgr_data);
                if (!$ssgr_res) {
                    return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
                }
            } else {
                // 销出
                // 先进先出，先进入的商品，就要先出去，所以销出的时候，数据从前往后查
                // 查询当前仓库、仓库类别、出库商品的结余库存数量大于0的记录数据
                $now_ssgr_data = DB::table('stock_sell_goods_record')
                        ->where('ssgr_cid', $user_info->adm_cid)
                        ->where('ssgr_shop_id', $user_info->adm_shop_id)
                        ->where('ssgr_ware_id', $ware_id)
                        ->where('ssgr_waon_class', $waon_class)
                        ->where('ssgr_goo_id', $value)
                        ->where('ssgr_surplus_stock', '>', 0)
                        ->select('ssgr_id', 'ssgr_surplus_stock', 'ssgr_surplus_average_cost', 'ssgr_surplus_cost', 'ssgr_month_out_stock', 'ssgr_month_out_cost', 'ssgr_month_adjust_stock', 'ssgr_month_adjust_cost', 'ssgr_month_adjust_stock_type', 'ssgr_month_adjust_cost_type')
                        ->offset(0)->limit($goo_quantity[$value])
                        ->get();
                $finally_quantity = $goo_quantity[$value];//商品要出库的数量
                $finally_all_cost = 0;//本次出库的商品总金额
                foreach ($now_ssgr_data as $ssgr_out_key => $ssgr_out_value) {
                    //最后的结余库存 = 先进的库存数量 - 出库的数量
                    $finally_surplus_stock = $ssgr_out_value->ssgr_surplus_stock - $finally_quantity;
                    // 如果结余库存小于0，说明当前入库记录的库存数量不足以完全出库，需要把其他入库记录的库存继续出库
                    if ($finally_surplus_stock < 0) {
                        $finally_quantity = $finally_quantity - $ssgr_out_value->ssgr_surplus_stock;//剩余还需要出库的数量
                        //出库的成本 = 当前入库平均价 * 出库数量
                        $finally_cost = $ssgr_out_value->ssgr_surplus_average_cost * $ssgr_out_value->ssgr_surplus_stock;//当前出库的成本
                        $finally_all_cost += $finally_cost;//本次出库的商品总金额
                        $goo_all_price += $finally_cost;//已经全部出库的成本

                        $zero_ssgr_data['ssgr_surplus_stock'] = 0;
                        $zero_ssgr_data['ssgr_surplus_cost'] = 0;
                        $zero_ssgr_data['ssgr_month_out_stock'] = $ssgr_out_value->ssgr_surplus_stock;
                        $zero_ssgr_data['ssgr_month_out_cost'] = $ssgr_out_value->ssgr_surplus_average_cost * $ssgr_out_value->ssgr_surplus_stock;
                        $zero_ssgr_data['ssgr_month_adjust_stock'] = $ssgr_out_value->ssgr_surplus_stock;//月调库存数量
                        $zero_ssgr_data['ssgr_month_adjust_stock_type'] = $ssgr_out_value->ssgr_surplus_average_cost * $ssgr_out_value->ssgr_surplus_stock;//月调成本
                        $zero_ssgr_res = DB::table('stock_sell_goods_record')->where('ssgr_id', $ssgr_out_value->ssgr_id)->update($zero_ssgr_data);
                    } else {
                        // 如果结余库存不小于0，说明当前入库记录的库存数量足够完全出库
                        $finally_cost = $ssgr_out_value->ssgr_surplus_average_cost * $finally_quantity;//当前出库的成本
                        $finally_all_cost += $finally_cost;//本次出库的商品总金额
                        $goo_all_price += $finally_cost;//已经全部出库的成本

                        $zero_ssgr_data['ssgr_surplus_stock'] = $ssgr_out_value->ssgr_surplus_stock - $finally_quantity;//剩余结余库存数量
                        $zero_ssgr_data['ssgr_surplus_cost'] = $ssgr_out_value->ssgr_surplus_cost - $finally_cost;//剩余的结余成本
                        $zero_ssgr_data['ssgr_month_out_stock'] = $ssgr_out_value->ssgr_month_out_stock + $finally_quantity;//月出库存
                        $zero_ssgr_data['ssgr_month_out_cost'] = $ssgr_out_value->ssgr_month_out_stock + $finally_cost;//月出成本
                        $zero_ssgr_data['ssgr_month_adjust_stock'] = $ssgr_out_value->ssgr_month_adjust_stock + $finally_quantity;//月调库存数量
                        $zero_ssgr_data['ssgr_month_adjust_stock_type'] = $ssgr_out_value->ssgr_month_adjust_stock + $finally_cost;//月调成本
                        $zero_ssgr_res = DB::table('stock_sell_goods_record')->where('ssgr_id', $ssgr_out_value->ssgr_id)->update($zero_ssgr_data);
                        break;
                    }
                    if (!$zero_ssgr_res) {
                        return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
                    }
                }
            }

            // 进销存商品表
            // 查询商品的当前月份的商品进销存数据
            $now_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $now_year)
                    ->where('sssg_month', $now_month)
                    ->where('sssg_goo_id', $value)
                    ->select('sssg_id', 'sssg_surplus_stock', 'sssg_surplus_cost', 'sssg_month_enter_stock', 'sssg_month_enter_cost', 'sssg_month_out_stock', 'sssg_month_out_cost', 'sssg_month_adjust_stock', 'sssg_month_adjust_cost', 'sssg_month_adjust_stock_type', 'sssg_month_adjust_cost_type', 'sssg_month_out_stock_type', 'sssg_month_out_cost_type')
                    ->first();

            if (empty($now_sssg_data)) {
                // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
                // 获取最近一个月的期末成本，作为当前月份的期初成本
                $last_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $last_year)
                    // ->where('sssg_month', $last_month)
                    ->where('sssg_goo_id', $value)
                    ->orderBy('sssg_id','desc')
                    ->select('sssg_id', 'sssg_surplus_stock', 'sssg_surplus_cost')
                    ->first();
                $sssg_data['sssg_cid'] = $user_info->adm_cid;//公司id
                $sssg_data['sssg_superior_id'] = $shop_id;//所属总店id
                $sssg_data['sssg_shop_id'] = $user_info->adm_shop_id;//	所属门店id
                $sssg_data['sssg_ware_id'] = $ware_id;//仓库id
                $sssg_data['sssg_waon_class'] = $waon_class;//仓库类别
                $sssg_data['sssg_goo_id'] = $value;//商品id
                $sssg_data['sssg_year'] = $now_year;//年份
                $sssg_data['sssg_month'] = $now_month;//月
                $sssg_data['sssg_initial_stock'] = isset($last_sssg_data->sssg_surplus_stock) ? $last_sssg_data->sssg_surplus_stock : 0;//期初库存，总库存数量
                $sssg_data['sssg_initial_cost'] = isset($last_sssg_data->sssg_surplus_cost) ? $last_sssg_data->sssg_surplus_cost : 0;//期初金额-成本（单位：分）,总的成本
                if ($type == 1) {
                    // 采购
                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_data['sssg_initial_cost'] + $now_cost;//结余成本（单位：分），剩余总的成本
                    $sssg_data['sssg_month_enter_stock'] = $goo_quantity[$value];//月入库存数量
                    $sssg_data['sssg_month_enter_cost'] = $now_cost;//月入成本（单位：分）
                } elseif ($type == 2) {
                    // 销出
                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_data['sssg_initial_cost'] - $finally_all_cost;//结余成本（单位：分），剩余总的成本
                    $sssg_data['sssg_month_out_stock'] = $goo_quantity[$value];//月出库存数量
                    $sssg_data['sssg_month_out_cost'] = $finally_all_cost;//月出成本（单位：分）
                } elseif ($type == 3) {
                    // 调拨出库
                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_data['sssg_initial_cost'] - $finally_all_cost;//结余成本（单位：分），剩余总的成本
                    
                    $sssg_data['sssg_month_adjust_stock'] = $goo_quantity[$value];//月调库存数量
                    $sssg_data['sssg_month_adjust_stock_type'] = 1;//月调库存类型：1调出（页面显示负数）2调入（页面显示正数）
                    $sssg_data['sssg_month_adjust_cost'] = $finally_all_cost;//月出成本（单位：分）
                    $sssg_data['sssg_month_adjust_cost_type'] = 1;//月调成本类型：1负（页面显示负数）2正（页面显示正数）
                } elseif ($type == 4) {
                    // 调拨入库
                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_data['sssg_initial_cost'] - $finally_all_cost;//结余成本（单位：分），剩余总的成本
                    
                    $sssg_data['sssg_month_adjust_stock'] = $goo_quantity[$value];//月调库存数量
                    $sssg_data['sssg_month_adjust_stock_type'] = 2;//月调库存类型：1调出（页面显示负数）2调入（页面显示正数）
                    $sssg_data['sssg_month_adjust_cost'] = $finally_all_cost;//月出成本（单位：分）
                    $sssg_data['sssg_month_adjust_cost_type'] = 2;//月调成本类型：1负（页面显示负数）2正（页面显示正数）
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')->insert($sssg_data);
            } else {
                if ($type == 1) {
                    // 采购
                    // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost + $now_cost;//结余成本（单位：分），剩余总的成本
                    $sssg_data['sssg_month_enter_stock'] = $now_sssg_data->sssg_month_enter_stock + $goo_quantity[$value];//月入库存数量
                    if ($now_cost == '0') {
                        $sssg_data['sssg_month_enter_cost'] = 0;
                    } else {
                        $sssg_data['sssg_month_enter_cost'] = $now_sssg_data->sssg_month_enter_stock + $now_cost;//月入成本（单位：分）
                    }
                } elseif ($type == 2) {
                    // 销出
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost - $finally_all_cost;//结余成本（单位：分），剩余总的成本
                    //月出库存数量
                    if ($now_sssg_data->sssg_month_out_stock_type == 1) {
                        $sssg_data['sssg_month_out_stock'] = $now_sssg_data->sssg_month_out_stock + $goo_quantity[$value];//月出库存数量
                    }else{
                        $sssg_data['sssg_month_out_stock'] = $now_sssg_data->sssg_month_out_stock - $goo_quantity[$value];//月出库存数量
                        if($sssg_data['sssg_month_out_stock'] > 0){
                            $sssg_data['sssg_month_out_stock_type'] = 2;
                        }else{
                            $sssg_data['sssg_month_out_stock_type'] = 1;
                        }
                        $sssg_data['sssg_month_out_stock'] = abs($sssg_data['sssg_month_out_stock']);
                    }
                    //月出成本（单位：分）
                    if ($now_sssg_data->sssg_month_out_cost_type == 1) {
                        $sssg_data['sssg_month_out_cost'] = $now_sssg_data->sssg_month_out_cost + $now_cost;//月出成本（单位：分）
                    }else{
                        $sssg_data['sssg_month_out_cost'] = $now_sssg_data->sssg_month_out_cost - $now_cost;//月出成本（单位：分）
                        if($sssg_data['sssg_month_out_cost'] > 0){
                            $sssg_data['sssg_month_out_cost_type'] = 2;
                        }else{
                            $sssg_data['sssg_month_out_cost_type'] = 1;
                        }
                        $sssg_data['sssg_month_out_cost'] = abs($sssg_data['sssg_month_out_cost']);
                    }

                } elseif ($type == 3) {
                    // 调拨出库
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost - $finally_all_cost;//结余成本（单位：分），剩余总的成本
                    
                    $sssg_data['sssg_month_adjust_stock'] = $now_sssg_data->sssg_month_adjust_stock + $goo_quantity[$value];//月调库存数量
                    $sssg_data['sssg_month_adjust_stock'] = 1;//月调库存类型：1调出（页面显示负数）2调入（页面显示正数）
                    $sssg_data['sssg_month_out_cost'] = $now_sssg_data->sssg_month_out_cost + $finally_all_cost;//月出成本（单位：分）
                    $sssg_data['sssg_month_out_cost'] = 1;//月调成本类型：1负（页面显示负数）2正（页面显示正数）
                } elseif ($type == 4) {
                    // 调拨入库
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost - $finally_all_cost;//结余成本（单位：分），剩余总的成本
                    
                    $sssg_data['sssg_month_adjust_stock'] = $now_sssg_data->sssg_month_adjust_stock + $goo_quantity[$value];//月调库存数量
                    $sssg_data['sssg_month_adjust_stock'] =2;//月调库存类型：1调出（页面显示负数）2调入（页面显示正数）
                    $sssg_data['sssg_month_out_cost'] = $now_sssg_data->sssg_month_out_cost + $finally_all_cost;//月出成本（单位：分）
                    $sssg_data['sssg_month_out_cost'] = 2;//月调成本类型：1负（页面显示负数）2正（页面显示正数）
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')
                        ->where('sssg_id', $now_sssg_data->sssg_id)
                        ->update($sssg_data);
            }

            if (!$sssg_res) {
                return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
            }
        }

        // 进销存查询统计表
        // 查询当前月份、仓库、仓库类别的数据
        $now_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $now_year)
                ->where('stss_month', $now_month)
                ->select('stss_id', 'stss_surplus_stock', 'stss_surplus_cost', 'stss_month_enter_stock', 'stss_month_enter_cost', 'stss_month_out_stock', 'stss_month_out_cost')
                ->first();
        if (empty($now_stss_data)) {
            // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
            // 获取上月的期末成本，作为当前月份的期初成本
            $last_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $last_year)
                ->where('stss_month', $last_month)
                ->select('stss_id', 'stss_surplus_stock', 'stss_surplus_cost')
                ->first();
            $stss_data['stss_cid'] = $user_info->adm_cid;//公司id
            $stss_data['stss_superior_id'] = $shop_id;//所属总店id
            $stss_data['stss_shop_id'] = $user_info->adm_shop_id;//	所属门店id
            $stss_data['stss_ware_id'] = $ware_id;//仓库id
            $stss_data['stss_waon_class'] = $waon_class;//仓库类别
            $stss_data['stss_year'] = $now_year;//年份
            $stss_data['stss_month'] = $now_month;//月
            $stss_data['stss_initial_stock'] = isset($last_stss_data->stss_surplus_stock) ? $last_stss_data->stss_surplus_stock : 0;//期初库存，总库存数量
            $stss_data['stss_initial_cost'] = isset($last_stss_data->stss_surplus_cost) ? $last_stss_data->stss_surplus_cost : 0;//期初金额-成本（单位：分）,总的成本
            if ($type == 1) {
                // 采购
                $stss_data['stss_surplus_stock'] = $stss_data['stss_initial_stock'] + $goo_quantity[$value];//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $stss_data['stss_initial_cost'] + $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_month_enter_stock'] = $goo_quantity[$value];//月入库存数量
                $stss_data['stss_month_enter_cost'] = $goo_all_price;//月入成本（单位：分）
            } else {
                // 销出
                $stss_data['stss_surplus_stock'] = $stss_data['stss_initial_stock'] - $goo_quantity[$value];//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $stss_data['stss_initial_cost'] - $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_month_out_stock'] = $goo_quantity[$value];//月入库存数量
                $stss_data['stss_month_out_cost'] = $goo_all_price;//月入成本（单位：分）
            }
            $stss_data['stss_update_time'] = date('Y-m-d H:i:s');
            $stss_res = DB::table('stock_sell_storage')->insert($stss_data);
        } else {
            if ($type == 1) {
                // 采购
                // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $now_stss_data->stss_surplus_cost + $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_month_enter_stock'] = $now_stss_data->stss_month_enter_stock + $goo_quantity[$value];//月入库存数量
                $stss_data['stss_month_enter_cost'] = $now_stss_data->stss_month_enter_cost + $goo_all_price;//月入成本（单位：分）
            } else {
                // 销出
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock - $goo_quantity[$value];//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $now_stss_data->stss_surplus_cost - $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_month_out_stock'] = $now_stss_data->stss_month_out_stock + $goo_quantity[$value];//月入库存数量
                $stss_data['stss_month_out_cost'] = $now_stss_data->stss_month_out_cost + $goo_all_price;//月入成本（单位：分）
            }
            $stss_res = DB::table('stock_sell_storage')->where('stss_id', $now_stss_data->stss_id)->update($stss_data);
        }

        if ($stss_res) {
            return ['code'=>200, 'msg'=>'success', 'data'=>'进销存更新成功'];
        } else {
            return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
        }
    }

    /**
     * [movingWeightedAverage 移动加权平均法]
     * @param  [type] $user_info    [用户信息]
     * @param  [type] $shop_id      [总店id]
     * @param  [type] $type         [类型：1采购，2销出，3调拨出库，4调拨入库 5采购退货出库]
     * @param  [type] $ware_id      [仓库id]
     * @param  [type] $waon_class   [仓库类别]
     * @param  [type] $goo_id       [数组，商品id]
     * @param  [type] $goo_quantity [数组，商品数量；[商品id=>商品数量]]
     * @param  [type] $goo_price    [数组，商品金额，成本价(不乘以100)；[商品id=>商品金额]]
     * @param  [type] $now_year     [当前年份]
     * @param  [type] $now_month    [当前月]
     * @param  [type] $last_year    [上个月的年份]
     * @param  [type] $last_month   [上个月]
     * @return [type]               [description]
     */
    private static function movingWeightedAverage(
        $user_info,
        $shop_id,
        $type,
        $ware_id,
        $waon_class,
        $goo_id,
        $goo_quantity,
        $goo_price,
        $now_year,
        $now_month,
        $last_year,
        $last_month
    ) {
        $goo_all_amount = 0;//商品总数量
        $goo_all_price = 0;//商品总金额
        foreach ($goo_id as $key => $value) {
            // 查询商品的当前月份的商品进销存数据
            $now_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $now_year)
                    ->where('sssg_month', $now_month)
                    ->where('sssg_goo_id', $value)
                    ->select('sssg_id', 'sssg_surplus_stock', 'sssg_surplus_average_cost', 'sssg_surplus_cost', 
                        'sssg_month_enter_stock', 'sssg_month_enter_cost', 'sssg_month_out_stock', 'sssg_month_out_cost', 
                        'sssg_surplus_stock_calculate', 'sssg_surplus_cost_calculate', 'sssg_month_adjust_stock', 
                        'sssg_month_adjust_cost', 'sssg_month_adjust_stock_type', 'sssg_month_adjust_cost_type', 'sssg_month_out_stock_type', 'sssg_month_out_cost_type')
                    ->first();

            // 进销存商品表
            if (empty($now_sssg_data)) {
                // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
                // 获取最近一个月的期末成本，作为当前月份的期初成本
                $last_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $last_year)
                    // ->where('sssg_month', $last_month)
                    ->where('sssg_goo_id', $value)
                    ->orderBy('sssg_id','desc')
                    ->select('sssg_id', 'sssg_surplus_stock', 'sssg_surplus_average_cost', 'sssg_surplus_cost', 'sssg_surplus_stock_calculate', 'sssg_surplus_cost_calculate')
                    ->first();
                $sssg_data['sssg_cid'] = $user_info->adm_cid;//公司id
                $sssg_data['sssg_superior_id'] = $shop_id;//所属总店id
                $sssg_data['sssg_shop_id'] = $user_info->adm_shop_id;//	所属门店id
                $sssg_data['sssg_ware_id'] = $ware_id;//仓库id
                $sssg_data['sssg_waon_class'] = $waon_class;//仓库类别
                $sssg_data['sssg_goo_id'] = $value;//商品id
                $sssg_data['sssg_year'] = $now_year;//年份
                $sssg_data['sssg_month'] = $now_month;//月
                //期初库存，总库存数量 = 上个月的结余库存
                $sssg_data['sssg_initial_stock'] = isset($last_sssg_data->sssg_surplus_stock) ? $last_sssg_data->sssg_surplus_stock : 0;
                //期初平均价（单位：分） = 上个月的期末平均价
                $sssg_data['sssg_initial_average_cost'] = isset($last_sssg_data->sssg_surplus_average_cost) ? $last_sssg_data->sssg_surplus_average_cost : 0;
                //期初金额-成本（单位：分）,总的成本 = 上个月的结余成本
                $sssg_data['sssg_initial_cost'] = isset($last_sssg_data->sssg_surplus_cost) ? $last_sssg_data->sssg_surplus_cost : 0;

                $sssg_data['sssg_surplus_stock_calculate'] = isset($last_sssg_data->sssg_surplus_stock_calculate) ? $last_sssg_data->sssg_surplus_stock_calculate : 0;
                $sssg_data['sssg_surplus_cost_calculate'] = isset($last_sssg_data->sssg_surplus_cost_calculate) ? $last_sssg_data->sssg_surplus_cost_calculate : 0;

                // 采购
                if ($type == 1) {
                    $now_cost = $goo_quantity[$value] * $goo_price[$value] * 100;//本次收入商品金额s
                    $sssg_data['sssg_month_enter_stock'] = $goo_quantity[$value];//月入库存数量
                    $sssg_data['sssg_month_enter_cost'] = $now_cost;//月入成本（单位：分）

                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_data['sssg_initial_cost'] + $sssg_data['sssg_month_enter_cost'];//结余成本（单位：分），剩余总的成本

                    $sssg_data['sssg_surplus_stock_calculate'] = $sssg_data['sssg_surplus_stock_calculate'] + $goo_quantity[$value];//进销存结余库存，进销存计算使用
                    $sssg_data['sssg_surplus_cost_calculate'] = $sssg_data['sssg_surplus_cost_calculate'] + $sssg_data['sssg_month_enter_cost'];//进销存结余成本（单位：分），进销存计算使用
                    
                    // 移动加权平均单价 = (本次收入前结存商品金额+本次收入商品金额) / (本次收入前结存商品数量+本次收入商品数量 )
                    $sssg_data['sssg_surplus_average_cost'] = round(($sssg_data['sssg_surplus_cost_calculate'] / 100 / $sssg_data['sssg_surplus_stock_calculate']), 2) * 100;
                // 销出
                } elseif ($type == 2) {
                    $now_cost = $goo_quantity[$value] * $sssg_data['sssg_initial_average_cost'];//本次(销售)商品金额
                    $sssg_data['sssg_month_out_stock'] = $goo_quantity[$value];//月出库存数量
                    $sssg_data['sssg_month_out_cost'] = $now_cost;//月出成本（单位：分）

                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_data['sssg_initial_cost'] - $sssg_data['sssg_month_out_cost'];//结余成本（单位：分），剩余总的成本

                    $sssg_data['sssg_surplus_stock_calculate'] = $sssg_data['sssg_surplus_stock_calculate'] - $goo_quantity[$value];//进销存结余库存，进销存计算使用
                    $sssg_data['sssg_surplus_cost_calculate'] = $sssg_data['sssg_surplus_cost_calculate'] - $sssg_data['sssg_month_out_cost'];//进销存结余成本（单位：分），进销存计算使用

                // 调拨出库，结余库存数量减少，结余成本减少，平均价不变，月调数量减少，月调成本减少；进销存数量不变，进销存结余成本不变
                } elseif ($type == 3) {
                    $now_cost = $goo_quantity[$value] * $sssg_data['sssg_initial_average_cost'];//本次(销售)商品金额
                    $sssg_data['sssg_month_adjust_stock'] = $goo_quantity[$value];//月调库存数量
                    $sssg_data['sssg_month_adjust_cost'] = $now_cost;//月调成本（单位：分）
                    $sssg_data['sssg_month_adjust_stock_type'] = 2;//月调库存类型正负：1正值2负值
                    $sssg_data['sssg_month_adjust_cost_type'] = 2;//月调成本类型正负：1正值2负值

                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_data['sssg_initial_cost'] - $now_cost;//结余成本（单位：分），剩余总的成本

                // 调拨入库，结余库存数量增加，结余成本增加，平均价不变，月调数量增加，月调成本增加；进销存数量不变，进销存结余成本不变
                } elseif ($type == 4) {
                    $now_cost = $goo_quantity[$value] * $sssg_data['sssg_initial_average_cost'];//本次(销售)商品金额
                    $sssg_data['sssg_month_adjust_stock'] = $goo_quantity[$value];//月调库存数量
                    $sssg_data['sssg_month_adjust_cost'] = $now_cost;//月调成本（单位：分）
                    $sssg_data['sssg_month_adjust_stock_type'] = 1;//月调库存类型正负：1正值2负值
                    $sssg_data['sssg_month_adjust_cost_type'] = 1;//月调成本类型正负：1正值2负值

                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_data['sssg_initial_cost'] + $now_cost;//结余成本（单位：分），剩余总的成本
                } elseif ($type == 5) {//采购出库
                    $now_cost = $goo_quantity[$value] * $goo_price[$value] * 100;//本次收入商品金额
                    $sssg_data['sssg_month_enter_stock'] = $goo_quantity[$value];//月入库存数量
                    $sssg_data['sssg_month_enter_cost'] = $now_cost;//月入成本（单位：分）

                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_data['sssg_initial_cost'] - $sssg_data['sssg_month_enter_cost'];//结余成本（单位：分），剩余总的成本

                    $sssg_data['sssg_surplus_stock_calculate'] = $sssg_data['sssg_surplus_stock_calculate'] - $goo_quantity[$value];//进销存结余库存，进销存计算使用
                    $sssg_data['sssg_surplus_cost_calculate'] = $sssg_data['sssg_surplus_cost_calculate'] - $sssg_data['sssg_month_enter_cost'];//进销存结余成本（单位：分），进销存计算使用
                    
                    // 移动加权平均单价 = (本次收入前结存商品金额+本次收入商品金额) / (本次收入前结存商品数量+本次收入商品数量 )
                    $sssg_data['sssg_surplus_average_cost'] = round(($sssg_data['sssg_surplus_cost_calculate'] / 100 / $sssg_data['sssg_surplus_stock_calculate']), 2) * 100;
                } else {
                    return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')->insert($sssg_data);
            } else {
                // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                // 采购
                if ($type == 1) {
                    $now_cost = $goo_quantity[$value] * $goo_price[$value] * 100;//本次收入商品金额
                    $sssg_data['sssg_month_enter_stock'] = $now_sssg_data->sssg_month_enter_stock + $goo_quantity[$value];//月入库存数量
                    $sssg_data['sssg_month_enter_cost'] = $now_sssg_data->sssg_month_enter_cost + $now_cost;//月入成本（单位：分）
                    $sssg_data['sssg_surplus_stock_calculate'] = $now_sssg_data->sssg_surplus_stock_calculate + $goo_quantity[$value];//进销存结余库存，进销存计算平均价使用
                    $sssg_data['sssg_surplus_cost_calculate'] = $now_sssg_data->sssg_surplus_cost_calculate + $now_cost;//进销存结余成本（单位：分），进销存计算平均价使用

                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost + $now_cost;//结余成本（单位：分），剩余总的成本
                    // 移动加权平均单价 = (本次收入前结存商品金额+本次收入商品金额) / (本次收入前结存商品数量+本次收入商品数量 )
                    // 四舍五入，保留两位小数，采购入库会重新计算平均价，销出不计算平均价
                    $sssg_data['sssg_surplus_average_cost'] = round(($sssg_data['sssg_surplus_cost_calculate'] / 100 / $sssg_data['sssg_surplus_stock_calculate']), 2) * 100;
                // 销出
                } elseif($type == 2) {
                    // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                    $now_cost = $goo_quantity[$value] * $now_sssg_data->sssg_surplus_average_cost;//本次(销售)商品金额
                    //月出库存数量
                    if ($now_sssg_data->sssg_month_out_stock_type == 1) {
                        $sssg_data['sssg_month_out_stock'] = $now_sssg_data->sssg_month_out_stock + $goo_quantity[$value];//月出库存数量
                    }else{
                        $sssg_data['sssg_month_out_stock'] = $now_sssg_data->sssg_month_out_stock - $goo_quantity[$value];//月出库存数量
                        if($sssg_data['sssg_month_out_stock'] > 0){
                            $sssg_data['sssg_month_out_stock_type'] = 2;
                        }else{
                            $sssg_data['sssg_month_out_stock_type'] = 1;
                        }
                        $sssg_data['sssg_month_out_stock'] = abs($sssg_data['sssg_month_out_stock']);
                    }
                    //月出成本（单位：分）
                    if ($now_sssg_data->sssg_month_out_cost_type == 1) {
                        $sssg_data['sssg_month_out_cost'] = $now_sssg_data->sssg_month_out_cost + $now_cost;//月出成本（单位：分）
                    }else{
                        $sssg_data['sssg_month_out_cost'] = $now_sssg_data->sssg_month_out_cost - $now_cost;//月出成本（单位：分）
                        if($sssg_data['sssg_month_out_cost'] > 0){
                            $sssg_data['sssg_month_out_cost_type'] = 2;
                        }else{
                            $sssg_data['sssg_month_out_cost_type'] = 1;
                        }
                        $sssg_data['sssg_month_out_cost'] = abs($sssg_data['sssg_month_out_cost']);
                    }
                    $sssg_data['sssg_surplus_stock_calculate'] = $now_sssg_data->sssg_surplus_stock_calculate - $goo_quantity[$value];//进销存结余库存，进销存计算平均价使用
                    $sssg_data['sssg_surplus_cost_calculate'] = $now_sssg_data->sssg_surplus_cost_calculate - $now_cost;//进销存结余成本（单位：分），进销存计算平均价使用

                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost - $now_cost;//结余成本（单位：分），剩余总的成本

                // 调拨出库，结余库存数量减少，结余成本减少，平均价不变，月调数量减少，月调成本减少；进销存数量不变，进销存结余成本不变
                } elseif ($type == 3) {
                    // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                    $now_cost = $goo_quantity[$value] * $now_sssg_data->sssg_surplus_average_cost;//本次(销售)商品金额

                    // 之前的月调库存是负值
                    if ($now_sssg_data->sssg_month_adjust_stock_type == 2) {
                        $sssg_data['sssg_month_adjust_stock'] = $now_sssg_data->sssg_month_adjust_stock + $goo_quantity[$value];
                        $sssg_data['sssg_month_adjust_stock_type'] = 2;//月调库存类型正负：1正值2负值
                    } else {
                        // 之前的月调库存是正值，减去现在的调拨出库的库存
                        $sssg_data['sssg_month_adjust_stock'] = $now_sssg_data->sssg_month_adjust_stock - $goo_quantity[$value];
                        if ($sssg_data['sssg_month_adjust_stock'] < 0) {
                            $sssg_data['sssg_month_adjust_stock_type'] = 2;//月调库存类型正负：1正值2负值
                        }else{
                            $sssg_data['sssg_month_adjust_stock_type'] = 1;//月调库存类型正负：1正值2负值
                        }
                        $sssg_data['sssg_month_adjust_stock'] = abs($sssg_data['sssg_month_adjust_stock']);//月调库存数量
                    }

                    // 之前的月调成本是负值
                    if ($now_sssg_data->sssg_month_adjust_cost_type == 2) {
                        $sssg_data['sssg_month_adjust_cost'] = $now_sssg_data->sssg_month_adjust_cost + $now_cost;
                        $sssg_data['sssg_month_adjust_stock_type'] = 2;//月调库存类型正负：1正值2负值
                    } else {
                        // 之前的月调成本是正值，减去现在的调拨出库的成本
                        $sssg_data['sssg_month_adjust_cost'] = $now_sssg_data->sssg_month_adjust_cost - $now_cost;
                        if ($sssg_data['sssg_month_adjust_cost'] < 0) {
                            $sssg_data['sssg_month_adjust_stock_type'] = 2;//月调库存类型正负：1正值2负值
                        }else{
                            $sssg_data['sssg_month_adjust_stock_type'] = 1;//月调库存类型正负：1正值2负值
                        }
                        $sssg_data['sssg_month_adjust_cost'] = abs($sssg_data['sssg_month_adjust_cost']);//月调成本（单位：分）
                    }

                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost - $now_cost;//结余成本（单位：分），剩余总的成本
                // 调拨入库，结余库存数量增加，结余成本增加，平均价不变，月调数量增加，月调成本增加；进销存数量不变，进销存结余成本不变
                } elseif ($type == 4) {
                    // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                    $now_cost = $goo_quantity[$value] * $now_sssg_data->sssg_surplus_average_cost;//本次(销售)商品金额

                    // 之前的月调库存是负值
                    if ($now_sssg_data->sssg_month_adjust_stock_type == 1) {
                        $sssg_data['sssg_month_adjust_stock'] = $now_sssg_data->sssg_month_adjust_stock + $goo_quantity[$value];
                        $sssg_data['sssg_month_adjust_stock_type'] = 1;//月调库存类型正负：1正值2负值
                    } else {
                        // 之前的月调库存是正值，减去现在的调拨出库的库存
                        $sssg_data['sssg_month_adjust_stock'] = $goo_quantity[$value] - $now_sssg_data->sssg_month_adjust_stock;
                        if ($sssg_data['sssg_month_adjust_stock'] < 0) {
                            $sssg_data['sssg_month_adjust_stock_type'] = 2;//月调库存类型正负：1正值2负值
                        }else{
                            $sssg_data['sssg_month_adjust_stock_type'] = 1;//月调库存类型正负：1正值2负值
                        }
                        $sssg_data['sssg_month_adjust_stock'] = abs($sssg_data['sssg_month_adjust_stock']);//月调库存数量
                    }

                    // 之前的月调成本是正值
                    if ($now_sssg_data->sssg_month_adjust_cost_type == 1) {
                        $sssg_data['sssg_month_adjust_cost'] = $now_sssg_data->sssg_month_adjust_cost + $now_cost;
                        $sssg_data['sssg_month_adjust_stock_type'] = 1;//月调库存类型正负：1正值2负值
                    } else {
                        // 之前的月调成本是负值
                        $sssg_data['sssg_month_adjust_cost'] = $now_cost - $now_sssg_data->sssg_month_adjust_cost;
                        if ($sssg_data['sssg_month_adjust_cost'] < 0) {
                            $sssg_data['sssg_month_adjust_stock_type'] = 2;//月调库存类型正负：1正值2负值
                        }else{
                            $sssg_data['sssg_month_adjust_stock_type'] = 1;//月调库存类型正负：1正值2负值
                        }
                        $sssg_data['sssg_month_adjust_cost'] = abs($sssg_data['sssg_month_adjust_cost']);//月调成本（单位：分）
                    }

                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost - $now_cost;//结余成本（单位：分），剩余总的成本
                } elseif ($type == 5) {
                    $now_cost = $goo_quantity[$value] * $goo_price[$value] * 100;//本次收入商品金额
                    $sssg_data['sssg_month_enter_stock'] = $now_sssg_data->sssg_month_enter_stock - $goo_quantity[$value];//月入库存数量
                    $sssg_data['sssg_month_enter_cost'] = $now_sssg_data->sssg_month_enter_cost - $now_cost;//月入成本（单位：分）
                    $sssg_data['sssg_surplus_stock_calculate'] = $now_sssg_data->sssg_surplus_stock_calculate - $goo_quantity[$value];//进销存结余库存，进销存计算平均价使用
                    $sssg_data['sssg_surplus_cost_calculate'] = $now_sssg_data->sssg_surplus_cost_calculate - $now_cost;//进销存结余成本（单位：分），进销存计算平均价使用

                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost - $now_cost;//结余成本（单位：分），剩余总的成本
                    // 移动加权平均单价 = (本次收入前结存商品金额+本次收入商品金额) / (本次收入前结存商品数量+本次收入商品数量 )
                    // 四舍五入，保留两位小数，采购入库会重新计算平均价，销出不计算平均价
                    if ($sssg_data['sssg_surplus_stock_calculate']) {
                        $sssg_data['sssg_surplus_average_cost'] = round(($sssg_data['sssg_surplus_cost_calculate'] / 100 / $sssg_data['sssg_surplus_stock_calculate']), 2) * 100;
                    }else{
                        $sssg_data['sssg_surplus_average_cost'] = 0;
                    }
                } else {
                    return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')
                        ->where('sssg_id', $now_sssg_data->sssg_id)
                        ->update($sssg_data);
            }

            $goo_all_amount += $goo_quantity[$value];//商品总数量
            $goo_all_price += $now_cost;//商品总金额
            if (!$sssg_res) {
                return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
            }
        }

        // 进销存查询统计表
        // 查询当前月份、仓库、仓库类别的数据
        $now_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $now_year)
                ->where('stss_month', $now_month)
                ->select('stss_id', 'stss_surplus_stock', 'stss_surplus_cost', 'stss_month_enter_stock', 'stss_month_enter_cost', 'stss_month_out_stock', 'stss_month_out_cost')
                ->first();
        if (empty($now_stss_data)) {
            // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
            // 获取上月的期末成本，作为当前月份的期初成本
            $last_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $last_year)
                ->where('stss_month', $last_month)
                ->select('stss_id', 'stss_surplus_stock', 'stss_surplus_cost')
                ->first();
            $stss_data['stss_cid'] = $user_info->adm_cid;//公司id
            $stss_data['stss_superior_id'] = $shop_id;//所属总店id
            $stss_data['stss_shop_id'] = $user_info->adm_shop_id;//	所属门店id
            $stss_data['stss_ware_id'] = $ware_id;//仓库id
            $stss_data['stss_waon_class'] = $waon_class;//仓库类别
            $stss_data['stss_year'] = $now_year;//年份
            $stss_data['stss_month'] = $now_month;//月
            $stss_data['stss_initial_stock'] = isset($last_stss_data->stss_surplus_stock) ? $last_stss_data->stss_surplus_stock : 0;//期初库存，总库存数量
            $stss_data['stss_initial_cost'] = isset($last_stss_data->stss_surplus_cost) ? $last_stss_data->stss_surplus_cost : 0;//期初金额-成本（单位：分）,总的成本
            if ($type == 1) {
                // 采购
                $stss_data['stss_surplus_stock'] = $stss_data['stss_initial_stock'] + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $stss_data['stss_initial_cost'] + $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_month_enter_stock'] = $goo_all_amount;//月入库存数量
                $stss_data['stss_month_enter_cost'] = $goo_all_price;//月入成本（单位：分）
            } else {
                // 销出
                $stss_data['stss_surplus_stock'] = $stss_data['stss_initial_stock'] - $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $stss_data['stss_initial_cost'] - $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_month_out_stock'] = $goo_all_amount;//月出库存数量
                $stss_data['stss_month_out_cost'] = $goo_all_price;//月出成本（单位：分）
            }
            $stss_data['stss_update_time'] = date('Y-m-d H:i:s');
            $stss_res = DB::table('stock_sell_storage')->insert($stss_data);
        } else {
            // 当前月份已经添加了商品的进销存，更改商品的进销存数据
            if ($type == 1) {
                // 采购
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $now_stss_data->stss_surplus_cost + $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_month_enter_stock'] = $now_stss_data->stss_month_enter_stock + $goo_all_amount;//月入库存数量
                $stss_data['stss_month_enter_cost'] = $now_stss_data->stss_month_enter_cost + $goo_all_price;//月入成本（单位：分）
            } else {
                // 销出
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock - $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $now_stss_data->stss_surplus_cost - $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_month_out_stock'] = $now_stss_data->stss_month_out_stock + $goo_all_amount;//月出库存数量
                $stss_data['stss_month_out_cost'] = $now_stss_data->stss_month_out_cost + $goo_all_price;//月出成本（单位：分）
            }
            $stss_data['stss_update_time'] = date('Y-m-d H:i:s');
            $stss_res = DB::table('stock_sell_storage')->where('stss_id', $now_stss_data->stss_id)->update($stss_data);
        }

        if ($stss_res) {
            return ['code'=>200, 'msg'=>'success', 'data'=>'进销存更新成功'];
        } else {
            return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
        }
    }

    /**
     * [noCost 无成本：只计算库存数量，不计算成本金额]
     * @param  [type] $user_info    [用户信息]
     * @param  [type] $shop_id      [总店id]
     * @param  [type] $type         [类型：1采购，2销出，3调拨出库，4调拨入库 5采购退货出库]
     * @param  [type] $ware_id      [仓库id]
     * @param  [type] $waon_class   [仓库类别]
     * @param  [type] $goo_id       [数组，商品id]
     * @param  [type] $goo_quantity [数组，商品数量；[商品id=>商品数量]]
     * @param  [type] $goo_price    [数组，商品金额，成本价(不乘以100)；[商品id=>商品金额]]
     * @param  [type] $now_year     [当前年份]
     * @param  [type] $now_month    [当前月]
     * @param  [type] $last_year    [上个月的年份]
     * @param  [type] $last_month   [上个月]
     * @return [type]               [description]
     */
    private static function noCost(
        $user_info,
        $shop_id,
        $type,
        $ware_id,
        $waon_class,
        $goo_id,
        $goo_quantity,
        $goo_price,
        $now_year,
        $now_month,
        $last_year,
        $last_month
    ) {
        $goo_all_amount = 0;//商品总数量
        foreach ($goo_id as $key => $value) {
            $goo_all_amount += $goo_quantity[$value];
            // 查询商品的当前月份的商品进销存数据
            $now_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $now_year)
                    ->where('sssg_month', $now_month)
                    ->where('sssg_goo_id', $goo_id)
                    ->select('sssg_id', 'sssg_surplus_stock', 'sssg_month_enter_stock', 'sssg_month_out_stock', 'sssg_month_adjust_stock', 'sssg_month_adjust_cost', 'sssg_month_adjust_stock_type', 'sssg_month_out_stock_type', 'sssg_month_out_cost_type')
                    ->first();
            // 进销存商品表
            if (empty($now_sssg_data)) {
                // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
                // 获取最近一个月的期末成本，作为当前月份的期初成本
                $last_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $last_year)
                    // ->where('sssg_month', $last_month)
                    ->where('sssg_goo_id', $goo_id)
                    ->orderBy('sssg_id','desc')
                    ->select('sssg_id', 'sssg_surplus_stock')
                    ->first();
                $sssg_data['sssg_cid'] = $user_info->adm_cid;//公司id
                $sssg_data['sssg_superior_id'] = $shop_id;//所属总店id
                $sssg_data['sssg_shop_id'] = $user_info->adm_shop_id;//	所属门店id
                $sssg_data['sssg_ware_id'] = $ware_id;//仓库id
                $sssg_data['sssg_waon_class'] = $waon_class;//仓库类别
                $sssg_data['sssg_goo_id'] = $value;//商品id
                $sssg_data['sssg_year'] = $now_year;//年份
                $sssg_data['sssg_month'] = $now_month;//月
                $sssg_data['sssg_initial_stock'] = isset($last_sssg_data->sssg_surplus_stock) ? $last_sssg_data->sssg_surplus_stock : 0;//期初库存，总库存数量
                if ($type == 1) {
                    // 采购
                    $sssg_data['sssg_month_enter_stock'] = $goo_quantity[$value];//月入库存数量
                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] + $goo_quantity[$value];//结余库存，剩余总库存数量
                } elseif ($type == 2) {
                    // 销出
                       // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_month_out_stock'] = $goo_quantity[$value];//月出库存数量
                } elseif ($type == 3) {
                    // 调出
                    // 调拨出库，结余库存数量减少
                    $sssg_data['sssg_month_adjust_stock'] = $goo_quantity[$value];//月调库存数量
                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_month_adjust_stock_type'] = 1;//月调类型：1调出（页面显示负数）2调入（页面显示正数）
                } elseif ($type==4) {
                    // 调入
                    // 调拨入库，结余库存数量增加
                    $sssg_data['sssg_month_adjust_stock'] = $goo_quantity[$value];//月调库存数量
                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_month_adjust_stock_type'] = 2;//月调类型：1调出（页面显示负数）2调入（页面显示正数）
                } elseif ($type == 5) {//采购退货出库
                    $sssg_data['sssg_month_enter_stock'] = $goo_quantity[$value];//月入库存数量
                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] - $goo_quantity[$value];//结余库存，剩余总库存数量
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')->insert($sssg_data);
            } else {
                if ($type == 1) {
                    // 采购
                    // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_month_enter_stock'] = $now_sssg_data->sssg_month_enter_stock + $goo_quantity[$value];//月入库存数量
                } elseif ($type == 2) {
                    // 销出
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock - $goo_quantity[$value];//结余库存，剩余总库存数量
                    if ($now_sssg_data->sssg_month_out_stock_type == 1) {
                        $sssg_data['sssg_month_out_stock'] = $now_sssg_data->sssg_month_out_stock + $goo_quantity[$value];//月出库存数量
                        $sssg_data['sssg_month_out_stock_type'] = 1;
                    }else{
                        $sssg_data['sssg_month_out_stock'] = $now_sssg_data->sssg_month_out_stock - $goo_quantity[$value];//月出库存数量
                        if($sssg_data['sssg_month_out_stock'] > 0){
                            $sssg_data['sssg_month_out_stock_type'] = 2;
                        }else{
                            $sssg_data['sssg_month_out_stock_type'] = 1;
                        }
                        $sssg_data['sssg_month_out_stock'] = abs($sssg_data['sssg_month_out_stock']);
                    }
                } elseif ($type == 3) {
                    // 调出
                    // 调拨出库，结余库存数量减少
                    $sssg_data['sssg_month_adjust_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//月调库存数量
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_month_enter_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_month_adjust_stock_type'] = 1;//月调类型：1调出（页面显示负数）2调入（页面显示正数）
                } elseif ($type==4) {
                    // 调入
                    // 调拨入库，结余库存数量增加
                    $sssg_data['sssg_month_adjust_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//月调库存数量
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_month_adjust_stock_type'] = 2;//月调类型：1调出（页面显示负数）2调入（页面显示正数）
                } elseif ($type == 5) {//采购退货出库
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_month_enter_stock'] = $now_sssg_data->sssg_month_enter_stock - $goo_quantity[$value];//月入库存数量
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')
                        ->where('sssg_id', $now_sssg_data->sssg_id)
                        ->update($sssg_data);
            }
            if (!$sssg_res) {
                return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败1'];
            }
        }

        // 进销存查询统计表
        // 查询当前月份、仓库、仓库类别的数据
        $now_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $now_year)
                ->where('stss_month', $now_month)
                ->select('stss_id', 'stss_surplus_stock', 'stss_month_enter_stock', 'stss_month_out_stock', 'stss_month_adjust_stock', 'stss_month_adjust_stock_type')
                ->first();
        if (empty($now_stss_data)) {
            // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
            // 获取上月的期末成本，作为当前月份的期初成本
            $last_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $last_year)
                ->where('stss_month', $last_month)
                ->select('stss_id', 'stss_surplus_stock')
                ->first();
            $stss_data['stss_cid'] = $user_info->adm_cid;//公司id
            $stss_data['stss_superior_id'] = $shop_id;//所属总店id
            $stss_data['stss_shop_id'] = $user_info->adm_shop_id;//	所属门店id
            $stss_data['stss_ware_id'] = $ware_id;//仓库id
            $stss_data['stss_waon_class'] = $waon_class;//仓库类别
            $stss_data['stss_year'] = $now_year;//年份
            $stss_data['stss_month'] = $now_month;//月
            $stss_data['stss_initial_stock'] = isset($last_stss_data->stss_surplus_stock) ? $last_stss_data->stss_surplus_stock : 0;//期初库存，总库存数量
            if ($type == 1) {
                // 采购
                $stss_data['stss_surplus_stock'] = $stss_data['stss_initial_stock'] + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_month_enter_stock'] = $goo_all_amount;//月入库存数量
            } elseif ($type == 2) {
                // 销出
                $stss_data['stss_surplus_stock'] = $stss_data['stss_initial_stock'] - $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_month_out_stock'] = $goo_all_amount;//月出库存数量
            } elseif ($type == 3) {
                // 调拨出库
                $stss_data['stss_surplus_stock'] = $stss_data['stss_initial_stock'] + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_month_adjust_stock'] = $goo_all_amount;//月调库存数量
                $sssg_data['stss_month_adjust_stock_type'] = 1;//月调类型：1调出（页面显示负数）2调入（页面显示正数）
            } elseif ($type == 4) {
                // 调拨入库
                $stss_data['stss_surplus_stock'] = $stss_data['stss_initial_stock'] + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_month_adjust_stock'] = $goo_all_amount;//月调库存数量
                $sssg_data['stss_month_adjust_stock_type'] = 2;//月调类型：1调出（页面显示负数）2调入（页面显示正数）
            } elseif ($type == 5) {
                $stss_data['stss_surplus_stock'] = $stss_data['stss_initial_stock'] - $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_month_enter_stock'] = $goo_all_amount;//月入库存数量
            }
            $stss_data['stss_update_time'] = date('Y-m-d H:i:s');
            $stss_res = DB::table('stock_sell_storage')->insert($stss_data);
        } else {
            // 当前月份已经添加了商品的进销存，更改商品的进销存数据
            if ($type == 1) {
                // 采购
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_month_enter_stock'] = $now_stss_data->stss_month_enter_stock + $goo_all_amount;//月入库存数量
            } elseif ($type == 2) {
                // 销出
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock - $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_month_out_stock'] = $now_stss_data->stss_month_out_stock + $goo_all_amount;//月出库存数量
            } elseif ($type == 3) {
                // 调拨出库
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_month_adjust_stock'] = $now_stss_data->stss_month_adjust_stock + $goo_all_amount;//月出库存数量
                $sssg_data['stss_month_adjust_stock_type'] = 1;//月调类型：1调出（页面显示负数）2调入（页面显示正数）
            } elseif ($type == 4) {
                // 调拨入库
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_month_adjust_stock'] = $now_stss_data->stss_month_adjust_stock + $goo_all_amount;//月出库存数量
                $sssg_data['stss_month_adjust_stock_type'] = 2;//月调类型：1调出（页面显示负数）2调入（页面显示正数）
            } elseif ($type == 5) {
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock - $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_month_enter_stock'] = $now_stss_data->stss_month_enter_stock - $goo_all_amount;//月入库存数量
            }
            $stss_data['stss_update_time'] = date('Y-m-d H:i:s');
            $stss_res = DB::table('stock_sell_storage')->where('stss_id', $now_stss_data->stss_id)->update($stss_data);
        }

        if ($stss_res) {
            return ['code'=>200, 'msg'=>'success', 'data'=>'进销存更新成功'];
        } else {
            return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败2'];
        }
    }

    /**
     * [movingWeightedNoAverage 移动加权平均不计算平均价算法]
     * @param  [type] $user_info    [用户信息]
     * @param  [type] $shop_id      [总店id]
     * @param  [type] $type         [类型：1采购，2销出，3调拨出库，4调拨入库5采购退货出库]
     * @param  [type] $ware_id      [仓库id]
     * @param  [type] $waon_class   [仓库类别]
     * @param  [type] $goo_id       [数组，商品id]
     * @param  [type] $goo_quantity [数组，商品数量；[商品id=>商品数量]]
     * @param  [type] $goo_price    [数组，商品金额，成本价(不乘以100)；[商品id=>商品金额]]
     * @param  [type] $now_year     [当前年份]
     * @param  [type] $now_month    [当前月]
     * @param  [type] $last_year    [上个月的年份]
     * @param  [type] $last_month   [上个月]
     * @return [type]               [description]
     */
    private static function movingWeightedNoAverage(
        $user_info,
        $shop_id,
        $type,
        $ware_id,
        $waon_class,
        $goo_id,
        $goo_quantity,
        $goo_price,
        $now_year,
        $now_month,
        $last_year,
        $last_month
    ) {
        $goo_all_amount = 0;//商品总数量
        $goo_all_price = 0;//商品总金额
        foreach ($goo_id as $key => $value) {
            $now_cost = $goo_quantity[$value] * $goo_price[$value] * 100;//本次收入(销售)商品金额
            $goo_all_amount += $goo_quantity[$value];//商品总数量
            $goo_all_price += $now_cost;//商品总金额

            // 查询商品的当前月份的商品进销存数据
            $now_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $now_year)
                    ->where('sssg_month', $now_month)
                    ->where('sssg_goo_id', $value)
                    ->select('sssg_id', 'sssg_surplus_stock', 'sssg_surplus_cost', 'sssg_month_enter_stock', 'sssg_month_enter_cost', 'sssg_month_out_stock', 'sssg_month_out_cost', 'sssg_month_adjust_stock', 'sssg_month_adjust_cost', 'sssg_month_adjust_stock_type', 'sssg_month_adjust_cost_type', 'sssg_month_out_stock_type', 'sssg_month_out_cost_type','sssg_surplus_average_cost')
                    ->first();

            // 进销存商品表
            if (empty($now_sssg_data)) {
                // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
                // 获取最近一个月的期末成本，作为当前月份的期初成本
                $last_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $last_year)
                    ->where('sssg_goo_id', $value)
                    ->orderBy('sssg_id','desc')
                    ->select('sssg_id', 'sssg_surplus_stock', 'sssg_surplus_average_cost', 'sssg_surplus_cost')
                    ->first();
                $sssg_data['sssg_cid'] = $user_info->adm_cid;//公司id
                $sssg_data['sssg_superior_id'] = $shop_id;//所属总店id
                $sssg_data['sssg_shop_id'] = $user_info->adm_shop_id;//	所属门店id
                $sssg_data['sssg_ware_id'] = $ware_id;//仓库id
                $sssg_data['sssg_waon_class'] = $waon_class;//仓库类别
                $sssg_data['sssg_goo_id'] = $value;//商品id
                $sssg_data['sssg_year'] = $now_year;//年份
                $sssg_data['sssg_month'] = $now_month;//月
                //期初库存，总库存数量 = 上个月的结余库存
                $sssg_data['sssg_initial_stock'] = isset($last_sssg_data->sssg_surplus_stock) ? $last_sssg_data->sssg_surplus_stock : 0;
                //期初平均价（单位：分） = 上个月的期末平均价
                $sssg_data['sssg_initial_average_cost'] = isset($last_sssg_data->sssg_surplus_average_cost) ? $last_sssg_data->sssg_surplus_average_cost : 0;
                //期初金额-成本（单位：分）,总的成本 = 上个月的结余成本
                $sssg_data['sssg_initial_cost'] = isset($last_sssg_data->sssg_surplus_cost) ? $last_sssg_data->sssg_surplus_cost : 0;
                if ($type == 1) {
                    // 采购
                    $sssg_data['sssg_month_enter_stock'] = $goo_quantity[$value];//月入库存数量
                    $sssg_data['sssg_month_enter_cost'] = $now_cost;//月入成本（单位：分）

                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_data['sssg_initial_cost'] + $sssg_data['sssg_month_enter_cost'];//结余成本（单位：分），剩余总的成本
                    // 移动加权平均单价 = 上次的平均价
                    $sssg_data['sssg_surplus_average_cost'] = $sssg_data['sssg_initial_average_cost'];
                } elseif ($type == 2) {
                    //成本拿上个月的期末平均价
                    $now_cost = $last_sssg_data->sssg_surplus_average_cost??0;
                    // 销出
                    $sssg_data['sssg_month_out_stock'] = $goo_quantity[$value];//月出库存数量
                    $sssg_data['sssg_month_out_cost'] = $now_cost;//月出成本（单位：分）

                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_data['sssg_initial_cost'] + $sssg_data['sssg_month_out_cost'];//结余成本（单位：分），剩余总的成本
                    $sssg_data['sssg_month_out_cost_type'] = 2;//月出成本正负1正值2负值
                    $sssg_data['sssg_month_out_stock_type'] = 2;//月出库存数量正负1正值2负值
                    $sssg_data['sssg_surplus_average_cost'] = $last_sssg_data->sssg_surplus_average_cost??0;//最近一次的期末平均价
                } elseif ($type == 3) {
                    // 调拨出库
                    $sssg_data['sssg_month_adjust_stock'] = $goo_quantity[$value];//月调库存数量
                    $sssg_data['sssg_month_adjust_stock_type'] = 1;//月调库存类型：1调出（页面显示负数）2调入（页面显示正数）
                    $sssg_data['sssg_month_adjust_cost'] = $now_cost;//月调成本（单位：分）
                    $sssg_data['sssg_month_adjust_cost_type'] = 1;//月调成本类型：1负（页面显示负数）2正（页面显示正数）

                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_data['sssg_initial_cost'] - $sssg_data['sssg_month_adjust_cost'];//结余成本（单位：分），剩余总的成本
                } elseif ($type == 4) {
                    // 调拨入库
                    $sssg_data['sssg_month_adjust_stock'] = $goo_quantity[$value];//月调库存数量
                    $sssg_data['sssg_month_adjust_stock_type'] = 2;//月调库存类型：1调出（页面显示负数）2调入（页面显示正数）
                    $sssg_data['sssg_month_adjust_cost'] = $now_cost;//月调成本（单位：分）
                    $sssg_data['sssg_month_adjust_cost_type'] = 2;//月调成本类型：1负（页面显示负数）2正（页面显示正数）

                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_data['sssg_initial_cost'] - $sssg_data['sssg_month_adjust_cost'];//结余成本（单位：分），剩余总的成本
                } elseif ($type == 5) {//采购退货出库
                    $sssg_data['sssg_month_enter_stock'] = $goo_quantity[$value];//月入库存数量
                    $sssg_data['sssg_month_enter_cost'] = $now_cost;//月入成本（单位：分）

                    $sssg_data['sssg_surplus_stock'] = $sssg_data['sssg_initial_stock'] - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_data['sssg_initial_cost'] - $sssg_data['sssg_month_enter_cost'];//结余成本（单位：分），剩余总的成本
                    // 移动加权平均单价 = 上次的平均价
                    $sssg_data['sssg_surplus_average_cost'] = $sssg_data['sssg_initial_average_cost'];
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')->insert($sssg_data);
            } else {
                // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                if ($type == 1) {
                    // 采购
                    $sssg_data['sssg_month_enter_stock'] = $now_sssg_data->sssg_month_enter_stock + $goo_quantity[$value];//月入库存数量
                    $sssg_data['sssg_month_enter_cost'] = $now_sssg_data->sssg_month_enter_cost + $now_cost;//月入成本（单位：分）
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost + $now_cost;//结余成本（单位：分），剩余总的成本
                    // 不更改期末平均价
                } elseif ($type == 2) {
                    //拿期末平均价
                    $now_cost = $now_sssg_data->sssg_surplus_average_cost??0;
                    //月出库存数量负数
                    if ($now_sssg_data->sssg_month_out_stock_type == 2) {
                        $sssg_data['sssg_month_out_stock'] = $now_sssg_data->sssg_month_out_stock + $goo_quantity[$value];//月出库存数量
                        $sssg_data['sssg_month_out_stock_type'] = 2;
                    }else{
                        $sssg_data['sssg_month_out_stock'] = $now_sssg_data->sssg_month_out_stock - $goo_quantity[$value];//月出库存数量
                        if($sssg_data['sssg_month_out_stock'] < 0){
                            $sssg_data['sssg_month_out_stock_type'] = 2;
                        }else{
                            $sssg_data['sssg_month_out_stock_type'] = 1;
                        }
                        $sssg_data['sssg_month_out_stock'] = abs($sssg_data['sssg_month_out_stock']);
                    }
                    // 月出成本负数
                    if($now_sssg_data->sssg_month_out_cost_type == 2){
                        $sssg_data['sssg_month_out_cost'] = $now_sssg_data->sssg_month_out_cost + $now_cost;//月出成本（单位：分）
                        $sssg_data['sssg_month_out_cost_type'] = 2;//1正数2负数
                    }else{
                        $sssg_data['sssg_month_out_cost'] = $now_sssg_data->sssg_month_out_cost - $now_cost;//月出成本（单位：分）
                        if($sssg_data['sssg_month_out_cost'] < 0){
                            $sssg_data['sssg_month_out_cost_type'] = 2;//1正数2负数
                        }else{
                            $sssg_data['sssg_month_out_cost_type'] = 1;//1正数2负数
                        }
                        $sssg_data['sssg_month_out_cost'] = abs($sssg_data['sssg_month_out_cost']);
                    }
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost + $now_cost;//结余成本（单位：分），剩余总的成本
                } elseif ($type == 3) {
                    // 调拨出库
                    // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                    $sssg_data['sssg_month_adjust_stock'] = $now_sssg_data->sssg_month_adjust_stock + $goo_quantity[$value];//月调库存数量
                    $sssg_data['sssg_month_adjust_stock_type'] = 1;//月调库存类型：1调出（页面显示负数）2调入（页面显示正数）
                    $sssg_data['sssg_month_adjust_cost'] = $now_sssg_data->sssg_month_adjust_cost + $now_cost;//月调成本（单位：分）
                    $sssg_data['sssg_month_adjust_cost_type'] = 1;//月调成本类型：1负（页面显示负数）2正（页面显示正数）
                    
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost - $now_cost;//结余成本（单位：分），剩余总的成本
                } elseif ($type == 4) {
                    // 调拨入库
                    // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                    $sssg_data['sssg_month_adjust_stock'] = $now_sssg_data->sssg_month_adjust_stock + $goo_quantity[$value];//月调库存数量
                    $sssg_data['sssg_month_adjust_stock_type'] = 2;//月调库存类型：1调出（页面显示负数）2调入（页面显示正数）
                    $sssg_data['sssg_month_adjust_cost'] = $now_sssg_data->sssg_month_adjust_cost + $now_cost;//月调成本（单位：分）
                    $sssg_data['sssg_month_adjust_cost_type'] = 1;//月调成本类型：1负（页面显示负数）2正（页面显示正数）
                    
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost - $now_cost;//结余成本（单位：分），剩余总的成本
                } elseif ($type == 5) {//采购退货出库
                    $sssg_data['sssg_month_enter_stock'] = $now_sssg_data->sssg_month_enter_stock - $goo_quantity[$value];//月入库存数量
                    $sssg_data['sssg_month_enter_cost'] = $now_sssg_data->sssg_month_enter_cost - $now_cost;//月入成本（单位：分）
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock - $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost - $now_cost;//结余成本（单位：分），剩余总的成本
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')
                        ->where('sssg_id', $now_sssg_data->sssg_id)
                        ->update($sssg_data);
            }
            if (!$sssg_res) {
                return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
            }
        }
        // 进销存查询统计表
        // 查询当前月份、仓库、仓库类别的数据
        $now_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $now_year)
                ->where('stss_month', $now_month)
                ->select('stss_id', 'stss_surplus_stock', 'stss_surplus_cost', 'stss_month_enter_stock', 'stss_month_enter_cost', 'stss_month_out_stock', 'stss_month_out_cost', 'stss_month_adjust_stock', 'stss_month_adjust_cost', 'stss_month_adjust_stock_type', 'stss_month_adjust_cost_type')
                ->first();
        if (empty($now_stss_data)) {
            // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
            // 获取上月的期末成本，作为当前月份的期初成本
            $last_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $last_year)
                ->where('stss_month', $last_month)
                ->select('stss_id', 'stss_surplus_stock', 'stss_surplus_cost')
                ->first();
            $stss_data['stss_cid'] = $user_info->adm_cid;//公司id
            $stss_data['stss_superior_id'] = $shop_id;//所属总店id
            $stss_data['stss_shop_id'] = $user_info->adm_shop_id;//	所属门店id
            $stss_data['stss_ware_id'] = $ware_id;//仓库id
            $stss_data['stss_waon_class'] = $waon_class;//仓库类别
            $stss_data['stss_year'] = $now_year;//年份
            $stss_data['stss_month'] = $now_month;//月
            $stss_data['stss_initial_stock'] = isset($last_stss_data->stss_surplus_stock) ? $last_stss_data->stss_surplus_stock : 0;//期初库存，总库存数量
            $stss_data['stss_initial_cost'] = isset($last_stss_data->stss_surplus_cost) ? $last_stss_data->stss_surplus_cost : 0;//期初金额-成本（单位：分）,总的成本
            if ($type == 1) {
                // 采购
                $stss_data['stss_surplus_stock'] = $stss_data['stss_initial_stock'] + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $stss_data['stss_initial_cost'] + $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_month_enter_stock'] = $goo_all_amount;//月入库存数量
                $stss_data['stss_month_enter_cost'] = $goo_all_price;//月入成本（单位：分）
            } elseif ($type == 2) {
                // 销出
                $stss_data['stss_surplus_stock'] = $stss_data['stss_initial_stock'] - $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $stss_data['stss_initial_cost'] - $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_month_out_stock'] = $goo_all_amount;//月出库存数量
                $stss_data['stss_month_out_cost'] = $goo_all_price;//月出成本（单位：分）
            } elseif ($type == 3) {
                // 调出
                $stss_data['stss_surplus_stock'] = $stss_data['stss_initial_stock'] - $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $stss_data['stss_initial_cost'] - $goo_all_price;//结余成本（单位：分），剩余总的成本
                
                $stss_data['stss_month_adjust_stock'] = $goo_all_amount;//月调库存数量
                $stss_data['stss_month_adjust_stock_type'] = 1;//月调库存类型：1调出（页面显示负数）2调入（页面显示正数）
                $stss_data['stss_month_adjust_cost'] = $goo_all_price;//月调成本（单位：分）
                $stss_data['stss_month_adjust_cost_type'] = 1;//月调成本类型：1负（页面显示负数）2正（页面显示正数）
            } elseif ($type == 4) {
                // 调入
                $stss_data['stss_surplus_stock'] = $stss_data['stss_initial_stock'] - $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $stss_data['stss_initial_cost'] - $goo_all_price;//结余成本（单位：分），剩余总的成本
                
                $stss_data['stss_month_adjust_stock'] = $goo_all_amount;//月调库存数量
                $stss_data['stss_month_adjust_stock_type'] = 2;//月调库存类型：1调出（页面显示负数）2调入（页面显示正数）
                $stss_data['stss_month_adjust_cost'] = $goo_all_price;//月调成本（单位：分）
                $stss_data['stss_month_adjust_cost_type'] = 2;//月调成本类型：1负（页面显示负数）2正（页面显示正数）
            } elseif ($type == 5) {//采购退货出库
                $stss_data['stss_surplus_stock'] = $stss_data['stss_initial_stock'] - $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $stss_data['stss_initial_cost'] - $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_month_enter_stock'] = $goo_all_amount;//月入库存数量
                $stss_data['stss_month_enter_cost'] = $goo_all_price;//月入成本（单位：分）
            }
            $stss_data['stss_update_time'] = date('Y-m-d H:i:s');
            $stss_res = DB::table('stock_sell_storage')->insert($stss_data);
        } else {
            // 当前月份已经添加了商品的进销存，更改商品的进销存数据
            if ($type == 1) {
                // 采购
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $now_stss_data->stss_surplus_cost + $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_month_enter_stock'] = $now_stss_data->stss_month_enter_stock + $goo_all_amount;//月入库存数量
                $stss_data['stss_month_enter_cost'] = $now_stss_data->stss_month_enter_cost + $goo_all_price;//月入成本（单位：分）
            } elseif ($type == 2) {
                // 销出
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock - $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $now_stss_data->stss_surplus_cost - $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_month_out_stock'] = $now_stss_data->stss_month_out_stock + $goo_all_amount;//月出库存数量
                $stss_data['stss_month_out_cost'] = $now_stss_data->stss_month_out_cost + $goo_all_price;//月出成本（单位：分）
            } elseif ($type == 3) {
                // 调拨出库
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock - $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $now_stss_data->stss_surplus_cost - $goo_all_price;//结余成本（单位：分），剩余总的成本
                
                $stss_data['stss_month_adjust_stock'] = $now_stss_data->stss_month_adjust_stock - $goo_all_amount;//月调库存数量
                $stss_data['stss_month_adjust_stock_type'] = 1;//月调库存类型：1调出（页面显示负数）2调入（页面显示正数）
                $stss_data['stss_month_adjust_cost'] = $now_stss_data->stss_month_adjust_cost - $goo_all_price;//月调成本（单位：分）
                $stss_data['stss_month_adjust_cost_type'] = 1;//月调成本类型：1负（页面显示负数）2正（页面显示正数）
            } elseif ($type == 4) {
                // 调拨入库
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock - $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $now_stss_data->stss_surplus_cost - $goo_all_price;//结余成本（单位：分），剩余总的成本
                
                $stss_data['stss_month_adjust_stock'] = $now_stss_data->stss_month_adjust_stock - $goo_all_amount;//月调库存数量
                $stss_data['stss_month_adjust_stock_type'] = 2;//月调库存类型：1调出（页面显示负数）2调入（页面显示正数）
                $stss_data['stss_month_adjust_cost'] = $now_stss_data->stss_month_adjust_cost - $goo_all_price;//月调成本（单位：分）
                $stss_data['stss_month_adjust_cost_type'] = 2;//月调成本类型：1负（页面显示负数）2正（页面显示正数）
            } elseif ($type == 5) {//采购退货出库
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock - $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $now_stss_data->stss_surplus_cost - $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_month_enter_stock'] = $now_stss_data->stss_month_enter_stock - $goo_all_amount;//月入库存数量
                $stss_data['stss_month_enter_cost'] = $now_stss_data->stss_month_enter_cost - $goo_all_price;//月入成本（单位：分）
            }
            $stss_data['stss_update_time'] = date('Y-m-d H:i:s');
            $stss_res = DB::table('stock_sell_storage')->where('stss_id', $now_stss_data->stss_id)->update($stss_data);
        }

        if ($stss_res) {
            return ['code'=>200, 'msg'=>'success', 'data'=>'进销存更新成功'];
        } else {
            return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
        }
    }

    /**
     * [costPrice 获取当前月份商品成本价]
     * @param  [type] $ware_id    [仓库id]
     * @param  [type] $waon_class [仓库类别]
     * @param  [type] $good_id    [商品id]
     * @return [type]             [description]
     */
    public static function costPrice($ware_id, $waon_class, $good_id)
    {
        $year = date('Y');
        $month = date('m');
        $res = DB::table('stock_sell_storage_goods')
            ->where('sssg_ware_id', $ware_id)
            ->where('sssg_waon_class', $waon_class)
            ->where('sssg_goo_id', $good_id)
            ->where('sssg_year', $year)
            ->where('sssg_month', $month)
            ->value('sssg_surplus_average_cost');
        if (empty($res)) {
            return 0;
        }
        return $res;
    }
    
    /**
     * [purchaseSalesImport 商品导入，进销存成本计算]
     * @param  [type]  $user_info      [用户信息]
     * @param  [type]  $algorithm_type [进销存计算方式：1先进先出2移动加权平均法3无成本]
     * @param  [type]  $type           [类型：1采购，2销出，3调拨]
     * @param  [type]  $ware_id        [仓库id]
     * @param  [type]  $waon_class     [仓库类别]
     * @param  [type]  $goo_id         [数组，商品id]
     * @param  [type]  $goo_quantity   [数组，商品数量；[商品id=>商品数量]]
     * @param  [type]  $goo_price      [数组，商品金额，成本价(不乘以100)；[商品id=>商品金额]]
     * @param  integer $average_type   [进销存的移动加权平均法算法类型：1计算平均价2不计算平均价【退换回收时不计算平均价】]
     * @return [type]                  [description]
     */
    public static function purchaseSalesImport(
        $user_info,
        $algorithm_type,
        $type,
        $ware_id,
        $waon_class,
        $goo_id,
        $goo_quantity,
        $goo_price,
        $average_type=1
    ) {
        //门店id
        $shop_id = $user_info->shop_superior_id ? $user_info->shop_superior_id : $user_info->adm_shop_id;
        $now_year = date('Y');//当前年
        $now_month = date('m');//当前月

        $last_month = $now_month - 1;//上个月
        //如果当前月是一月份，上个月的数据就为0
        if ($last_month == 0) {
            // 上个月就是去年的12月
            $last_month = 12;
            $last_year = $now_year - 1;
        } else {
            $last_year = $now_year;
        }
        
        if ($algorithm_type == 2) {
            // 移动加权平均法
            if ($average_type == 1) {
                return self::movingWeightedAverageImport(
                    $user_info,
                    $shop_id,
                    $type,
                    $ware_id,
                    $waon_class,
                    $goo_id,
                    $goo_quantity,
                    $goo_price,
                    $now_year,
                    $now_month,
                    $last_year,
                    $last_month
                );
            } elseif ($average_type == 3) {
                return self::movingWeightedAverageImports(
                    $user_info,
                    $shop_id,
                    $type,
                    $ware_id,
                    $waon_class,
                    $goo_id,
                    $goo_quantity,
                    $goo_price,
                    $now_year,
                    $now_month,
                    $last_year,
                    $last_month
                );
            }
        // 无成本
        } elseif ($algorithm_type == 3) {
            // 无成本
            return self::noCostImport(
                $user_info,
                $shop_id,
                $type,
                $ware_id,
                $waon_class,
                $goo_id,
                $goo_quantity,
                $goo_price,
                $now_year,
                $now_month,
                $last_year,
                $last_month
            );
        // 先进先出，进销存商品记录表，记录商品数据
        } elseif ($algorithm_type == 1){
            return self::firstInFirstOutImport(
                $user_info,
                $shop_id,
                $type,
                $ware_id,
                $waon_class,
                $goo_id,
                $goo_quantity,
                $goo_price,
                $now_year,
                $now_month,
                $last_year,
                $last_month
            );
        }
    }

    /**
     * [movingWeightedAverageImport 移动加权平均法]
     * @param  [type] $user_info    [用户信息]
     * @param  [type] $shop_id      [总店id]
     * @param  [type] $type         [类型：1采购，2销出，3调拨]
     * @param  [type] $ware_id      [仓库id]
     * @param  [type] $waon_class   [仓库类别]
     * @param  [type] $goo_id       [数组，商品id]
     * @param  [type] $goo_quantity [数组，商品数量；[商品id=>商品数量]]
     * @param  [type] $goo_price    [数组，商品金额，成本价(不乘以100)；[商品id=>商品金额]]
     * @param  [type] $now_year     [当前年份]
     * @param  [type] $now_month    [当前月]
     * @param  [type] $last_year    [上个月的年份]
     * @param  [type] $last_month   [上个月]
     * @return [type]               [description]
     */
    private static function movingWeightedAverageImport(
        $user_info,
        $shop_id,
        $type,
        $ware_id,
        $waon_class,
        $goo_id,
        $goo_quantity,
        $goo_price,
        $now_year,
        $now_month,
        $last_year,
        $last_month
    ) {
        $goo_all_amount = 0;//商品总数量
        $goo_all_price = 0;//商品总金额
        foreach ($goo_id as $key => $value) {
            // 查询商品的当前月份的商品进销存数据
            $now_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $now_year)
                    ->where('sssg_month', $now_month)
                    ->where('sssg_goo_id', $value)
                    ->select('sssg_id', 'sssg_surplus_stock', 'sssg_surplus_average_cost', 'sssg_surplus_cost', 'sssg_month_enter_stock', 'sssg_month_enter_cost', 'sssg_month_out_stock', 'sssg_month_out_cost')
                    ->first();

            // 进销存商品表
            if (empty($now_sssg_data)) {
                // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
                // 获取上月的期末成本，作为当前月份的期初成本
                $last_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $last_year)
                    ->where('sssg_month', $last_month)
                    ->where('sssg_goo_id', $value)
                    ->select('sssg_id', 'sssg_surplus_stock', 'sssg_surplus_average_cost', 'sssg_surplus_cost')
                    ->first();
                $sssg_data['sssg_cid'] = $user_info->adm_cid;//公司id
                $sssg_data['sssg_superior_id'] = $shop_id;//所属总店id
                $sssg_data['sssg_shop_id'] = $user_info->adm_shop_id;//	所属门店id
                $sssg_data['sssg_ware_id'] = $ware_id;//仓库id
                $sssg_data['sssg_waon_class'] = $waon_class;//仓库类别
                $sssg_data['sssg_goo_id'] = $value;//商品id
                $sssg_data['sssg_year'] = $now_year;//年份
                $sssg_data['sssg_month'] = $now_month;//月
                //期初库存，总库存数量 = 上个月的结余库存
                //期初平均价（单位：分） = 上个月的期末平均价
                $sssg_initial_stock = isset($last_sssg_data->sssg_surplus_stock) ? $last_sssg_data->sssg_surplus_stock : 0;
                // $sssg_data['sssg_initial_stock'] = isset($last_sssg_data->sssg_surplus_stock) ? $last_sssg_data->sssg_surplus_stock : 0;
                // $sssg_data['sssg_initial_average_cost'] = isset($last_sssg_data->sssg_surplus_average_cost) ? $last_sssg_data->sssg_surplus_average_cost : 0;
                //期初金额-成本（单位：分）,总的成本 = 上个月的结余成本
                $sssg_initial_cost = isset($last_sssg_data->sssg_surplus_cost) ? $last_sssg_data->sssg_surplus_cost : 0;
                // $sssg_data['sssg_initial_cost'] = isset($last_sssg_data->sssg_surplus_cost) ? $last_sssg_data->sssg_surplus_cost : 0;
                if ($type == 1) {
                    // 采购
                    $now_cost = $goo_quantity[$value] * $goo_price[$value] * 100;//本次收入商品金额s
                    // $sssg_data['sssg_month_enter_stock'] = $goo_quantity[$value];//月入库存数量
                    // $sssg_data['sssg_month_enter_cost'] = $now_cost;//月入成本（单位：分）

                    $sssg_data['sssg_initial_stock'] = $sssg_initial_stock + $goo_quantity[$value];//期初库存
                    $sssg_data['sssg_initial_cost'] = $sssg_initial_cost + $now_cost;//期初金额
                    
                    $sssg_data['sssg_surplus_stock'] = $sssg_initial_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_initial_cost + $now_cost;//结余成本（单位：分），剩余总的成本
                    // 移动加权平均单价 = (本次收入前结存商品金额+本次收入商品金额) / (本次收入前结存商品数量+本次收入商品数量 )
                    $sssg_data['sssg_initial_average_cost'] = round(($sssg_data['sssg_surplus_cost'] / 100 / $sssg_data['sssg_surplus_stock']), 2) * 100;
                    $sssg_data['sssg_surplus_average_cost'] = round(($sssg_data['sssg_surplus_cost'] / 100 / $sssg_data['sssg_surplus_stock']), 2) * 100;
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')->insert($sssg_data);
            } else {
                // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                if ($type == 1) {
                    // 采购
                    $now_cost = $goo_quantity[$value] * $goo_price[$value] * 100;//本次收入商品金额
                    // $sssg_data['sssg_month_enter_stock'] = $now_sssg_data->sssg_month_enter_stock + $goo_quantity[$value];//月入库存数量
                    // $sssg_data['sssg_month_enter_cost'] = $now_sssg_data->sssg_month_enter_cost + $now_cost;//月入成本（单位：分）

                    $sssg_data['sssg_initial_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//期初库存
                    $sssg_data['sssg_initial_cost'] = $now_sssg_data->sssg_surplus_cost + $now_cost;//期初金额
                    
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost + $now_cost;//结余成本（单位：分），剩余总的成本

                    // 移动加权平均单价 = (本次收入前结存商品金额+本次收入商品金额) / (本次收入前结存商品数量+本次收入商品数量 )
                    // 四舍五入，保留两位小数，采购入库会重新计算平均价，销出不计算平均价
                    $sssg_data['sssg_initial_average_cost'] = round(($sssg_data['sssg_surplus_cost'] / 100 / $sssg_data['sssg_surplus_stock']), 2) * 100;
                    $sssg_data['sssg_surplus_average_cost'] = round(($sssg_data['sssg_surplus_cost'] / 100 / $sssg_data['sssg_surplus_stock']), 2) * 100;
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')
                        ->where('sssg_id', $now_sssg_data->sssg_id)
                        ->update($sssg_data);
            }

            $goo_all_amount += $goo_quantity[$value];//商品总数量
            $goo_all_price += $now_cost;//商品总金额
            if (!$sssg_res) {
                return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
            }
        }

        // 进销存查询统计表
        // 查询当前月份、仓库、仓库类别的数据
        $now_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $now_year)
                ->where('stss_month', $now_month)
                ->select('stss_id', 'stss_surplus_stock', 'stss_surplus_cost', 'stss_month_enter_stock', 'stss_month_enter_cost', 'stss_month_out_stock', 'stss_month_out_cost')
                ->first();
        if (empty($now_stss_data)) {
            // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
            // 获取上月的期末成本，作为当前月份的期初成本
            $last_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $last_year)
                ->where('stss_month', $last_month)
                ->select('stss_id', 'stss_surplus_stock', 'stss_surplus_cost')
                ->first();
            $stss_data['stss_cid'] = $user_info->adm_cid;//公司id
            $stss_data['stss_superior_id'] = $shop_id;//所属总店id
            $stss_data['stss_shop_id'] = $user_info->adm_shop_id;//	所属门店id
            $stss_data['stss_ware_id'] = $ware_id;//仓库id
            $stss_data['stss_waon_class'] = $waon_class;//仓库类别
            $stss_data['stss_year'] = $now_year;//年份
            $stss_data['stss_month'] = $now_month;//月
            $stss_initial_stock = isset($last_stss_data->stss_surplus_stock) ? $last_stss_data->stss_surplus_stock : 0;//期初库存，总库存数量
            $stss_initial_cost = isset($last_stss_data->stss_surplus_cost) ? $last_stss_data->stss_surplus_cost : 0;//期初金额-成本（单位：分）,总的成本
            // $stss_data['stss_initial_stock'] = isset($last_stss_data->stss_surplus_stock) ? $last_stss_data->stss_surplus_stock : 0;//期初库存，总库存数量
            // $stss_data['stss_initial_cost'] = isset($last_stss_data->stss_surplus_cost) ? $last_stss_data->stss_surplus_cost : 0;//期初金额-成本（单位：分）,总的成本

            if ($type == 1) {
                // 采购
                $stss_data['stss_surplus_stock'] = $stss_initial_stock + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $stss_initial_cost + $goo_all_price;//结余成本（单位：分），剩余总的成本

                $stss_data['stss_initial_stock'] = $stss_initial_stock + $goo_all_amount;//期初库存
                $stss_data['stss_initial_cost'] = $stss_initial_cost + $goo_all_price;//期初金额

                // $stss_data['stss_month_enter_stock'] = $goo_all_amount;//月入库存数量
                // $stss_data['stss_month_enter_cost'] = $goo_all_price;//月入成本（单位：分）
            }
            $stss_data['stss_update_time'] = date('Y-m-d H:i:s');
            $stss_res = DB::table('stock_sell_storage')->insert($stss_data);
        } else {
            // 当前月份已经添加了商品的进销存，更改商品的进销存数据
            if ($type == 1) {
                // 采购
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $now_stss_data->stss_surplus_cost + $goo_all_price;//结余成本（单位：分），剩余总的成本

                $stss_data['stss_initial_stock'] = $now_stss_data->stss_surplus_stock + $goo_all_amount;//期初库存
                $stss_data['stss_initial_cost'] = $now_stss_data->stss_surplus_cost + $goo_all_price;//期初成本

                // $stss_data['stss_month_enter_stock'] = $now_stss_data->stss_month_enter_stock + $goo_all_amount;//月入库存数量
                // $stss_data['stss_month_enter_cost'] = $now_stss_data->stss_month_enter_cost + $goo_all_price;//月入成本（单位：分）
            }
            $stss_data['stss_update_time'] = date('Y-m-d H:i:s');
            $stss_res = DB::table('stock_sell_storage')->where('stss_id', $now_stss_data->stss_id)->update($stss_data);
        }

        if ($stss_res) {
            return ['code'=>200, 'msg'=>'success', 'data'=>'进销存更新成功'];
        } else {
            return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
        }
    }

    /**
     * [movingWeightedAverageImport 移动加权平均法]
     * @param  [type] $user_info    [用户信息]
     * @param  [type] $shop_id      [总店id]
     * @param  [type] $type         [类型：1采购，2销出，3调拨]
     * @param  [type] $ware_id      [仓库id]
     * @param  [type] $waon_class   [仓库类别]
     * @param  [type] $goo_id       [数组，商品id]
     * @param  [type] $goo_quantity [数组，商品数量；[商品id=>商品数量]]
     * @param  [type] $goo_price    [数组，商品金额，成本价(不乘以100)；[商品id=>商品金额]]
     * @param  [type] $now_year     [当前年份]
     * @param  [type] $now_month    [当前月]
     * @param  [type] $last_year    [上个月的年份]
     * @param  [type] $last_month   [上个月]
     * @return [type]               [description]
     */
    private static function movingWeightedAverageImports(
        $user_info,
        $shop_id,
        $type,
        $ware_id,
        $waon_class,
        $goo_id,
        $goo_quantity,
        $goo_price,
        $now_year,
        $now_month,
        $last_year,
        $last_month
    ) {
        $goo_all_amount = 0;//商品总数量
        $goo_all_price = 0;//商品总金额
        foreach ($goo_id as $key => $value) {
            // 查询商品的当前月份的商品进销存数据
            $now_sssg_data = DB::table('stock_sell_storage_goods')
                ->where('sssg_cid', $user_info->adm_cid)
                ->where('sssg_shop_id', $user_info->adm_shop_id)
                ->where('sssg_ware_id', $ware_id)
                ->where('sssg_waon_class', $waon_class)
                ->where('sssg_year', $now_year)
                ->where('sssg_month', $now_month)
                ->where('sssg_goo_id', $value)
                ->select('sssg_id', 'sssg_surplus_stock', 'sssg_surplus_average_cost', 'sssg_surplus_cost', 'sssg_month_enter_stock', 'sssg_month_enter_cost', 'sssg_month_out_stock', 'sssg_month_out_cost')
                ->first();

            // 进销存商品表
            if (empty($now_sssg_data)) {
                // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
                // 获取上月的期末成本，作为当前月份的期初成本
                $last_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $last_year)
                    ->where('sssg_month', $last_month)
                    ->where('sssg_goo_id', $value)
                    ->select('sssg_id', 'sssg_surplus_stock', 'sssg_surplus_average_cost', 'sssg_surplus_cost')
                    ->first();
                $sssg_data['sssg_cid'] = $user_info->adm_cid;//公司id
                $sssg_data['sssg_superior_id'] = $shop_id;//所属总店id
                $sssg_data['sssg_shop_id'] = $user_info->adm_shop_id;//	所属门店id
                $sssg_data['sssg_ware_id'] = $ware_id;//仓库id
                $sssg_data['sssg_waon_class'] = $waon_class;//仓库类别
                $sssg_data['sssg_goo_id'] = $value;//商品id
                $sssg_data['sssg_year'] = $now_year;//年份
                $sssg_data['sssg_month'] = $now_month;//月
                //期初库存，总库存数量 = 上个月的结余库存
                //期初平均价（单位：分） = 上个月的期末平均价
                $sssg_initial_stock = isset($last_sssg_data->sssg_surplus_stock) ? $last_sssg_data->sssg_surplus_stock : 0;
                //期初金额-成本（单位：分）,总的成本 = 上个月的结余成本
                $sssg_initial_cost = isset($last_sssg_data->sssg_surplus_cost) ? $last_sssg_data->sssg_surplus_cost : 0;
                if ($type == 1) {
                    // 采购
                    $now_cost = $goo_quantity[$value] * $goo_price[$value] * 100;//本次收入商品金额s

                    $sssg_data['sssg_initial_stock'] = $sssg_initial_stock + $goo_quantity[$value];//期初库存
                    $sssg_data['sssg_initial_cost'] = $sssg_initial_cost + $now_cost;//期初金额
                    
                    $sssg_data['sssg_surplus_stock'] = $sssg_initial_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_initial_cost + $now_cost;//结余成本（单位：分），剩余总的成本
                    // 移动加权平均单价 = (本次收入前结存商品金额+本次收入商品金额) / (本次收入前结存商品数量+本次收入商品数量 )
                    if ($sssg_data['sssg_surplus_stock'] == 0) {
                        $sssg_data['sssg_initial_average_cost'] = 0;
                    } else {
                        $sssg_data['sssg_initial_average_cost'] = round(($sssg_data['sssg_surplus_cost'] / 100 / $sssg_data['sssg_surplus_stock']), 2) * 100;
                    }
                    if ($sssg_data['sssg_surplus_stock'] == '0') {
                        $sssg_data['sssg_surplus_average_cost'] = 0;
                    } else {
                        $sssg_data['sssg_surplus_average_cost'] = round(($sssg_data['sssg_surplus_cost'] / 100 / $sssg_data['sssg_surplus_stock']), 2) * 100;
                    }
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')->insert($sssg_data);
            } else {
                // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                if ($type == 1) {
                    // 采购
                    $now_cost = $goo_quantity[$value] * $goo_price[$value] * 100;//本次收入商品金额

                    $sssg_data['sssg_initial_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//期初库存
                    $sssg_data['sssg_initial_cost'] = $now_sssg_data->sssg_surplus_cost + $now_cost;//期初金额
                    
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost + $now_cost;//结余成本（单位：分），剩余总的成本

                    // 移动加权平均单价 = (本次收入前结存商品金额+本次收入商品金额) / (本次收入前结存商品数量+本次收入商品数量 )
                    // 四舍五入，保留两位小数，采购入库会重新计算平均价，销出不计算平均价
                    if ($sssg_data['sssg_surplus_stock'] == '0') {
                        $sssg_data['sssg_initial_average_cost'] = 0;
                    } else {
                        $sssg_data['sssg_initial_average_cost'] = round(($sssg_data['sssg_surplus_cost'] / 100 / $sssg_data['sssg_surplus_stock']), 2) * 100;
                    }
                    if ($sssg_data['sssg_surplus_stock'] == '0') {
                        $sssg_data['sssg_surplus_average_cost'] = 0;
                    } else {
                        $sssg_data['sssg_surplus_average_cost'] = round(($sssg_data['sssg_surplus_cost'] / 100 / $sssg_data['sssg_surplus_stock']), 2) * 100;
                    }
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')
                        ->where('sssg_id', $now_sssg_data->sssg_id)
                        ->update($sssg_data);
            }

            $goo_all_amount += $goo_quantity[$value];//商品总数量
            $goo_all_price += $now_cost;//商品总金额
            if (!$sssg_res) {
                return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
            }
        }

        // 进销存查询统计表
        // 查询当前月份、仓库、仓库类别的数据
        $now_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $now_year)
                ->where('stss_month', $now_month)
                ->select('stss_id', 'stss_surplus_stock', 'stss_surplus_cost', 'stss_month_enter_stock', 'stss_month_enter_cost', 'stss_month_out_stock', 'stss_month_out_cost')
                ->first();
        if (empty($now_stss_data)) {
            // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
            // 获取上月的期末成本，作为当前月份的期初成本
            $last_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $last_year)
                ->where('stss_month', $last_month)
                ->select('stss_id', 'stss_surplus_stock', 'stss_surplus_cost')
                ->first();
            $stss_data['stss_cid'] = $user_info->adm_cid;//公司id
            $stss_data['stss_superior_id'] = $shop_id;//所属总店id
            $stss_data['stss_shop_id'] = $user_info->adm_shop_id;//	所属门店id
            $stss_data['stss_ware_id'] = $ware_id;//仓库id
            $stss_data['stss_waon_class'] = $waon_class;//仓库类别
            $stss_data['stss_year'] = $now_year;//年份
            $stss_data['stss_month'] = $now_month;//月
            $stss_initial_stock = isset($last_stss_data->stss_surplus_stock) ? $last_stss_data->stss_surplus_stock : 0;//期初库存，总库存数量
            $stss_initial_cost = isset($last_stss_data->stss_surplus_cost) ? $last_stss_data->stss_surplus_cost : 0;//期初金额-成本（单位：分）,总的成本
            // $stss_data['stss_initial_stock'] = isset($last_stss_data->stss_surplus_stock) ? $last_stss_data->stss_surplus_stock : 0;//期初库存，总库存数量
            // $stss_data['stss_initial_cost'] = isset($last_stss_data->stss_surplus_cost) ? $last_stss_data->stss_surplus_cost : 0;//期初金额-成本（单位：分）,总的成本

            if ($type == 1) {
                // 采购
                $stss_data['stss_surplus_stock'] = $stss_initial_stock + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $stss_initial_cost + $goo_all_price;//结余成本（单位：分），剩余总的成本

                $stss_data['stss_initial_stock'] = $stss_initial_stock + $goo_all_amount;//期初库存
                $stss_data['stss_initial_cost'] = $stss_initial_cost + $goo_all_price;//期初金额

                // $stss_data['stss_month_enter_stock'] = $goo_all_amount;//月入库存数量
                // $stss_data['stss_month_enter_cost'] = $goo_all_price;//月入成本（单位：分）
            }
            $stss_data['stss_update_time'] = date('Y-m-d H:i:s');
            $stss_res = DB::table('stock_sell_storage')->insert($stss_data);
        } else {
            // 当前月份已经添加了商品的进销存，更改商品的进销存数据
            if ($type == 1) {
                // 采购
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $now_stss_data->stss_surplus_cost + $goo_all_price;//结余成本（单位：分），剩余总的成本

                $stss_data['stss_initial_stock'] = $now_stss_data->stss_surplus_stock + $goo_all_amount;//期初库存
                $stss_data['stss_initial_cost'] = $now_stss_data->stss_surplus_cost + $goo_all_price;//期初成本

                // $stss_data['stss_month_enter_stock'] = $now_stss_data->stss_month_enter_stock + $goo_all_amount;//月入库存数量
                // $stss_data['stss_month_enter_cost'] = $now_stss_data->stss_month_enter_cost + $goo_all_price;//月入成本（单位：分）
            }
            $stss_data['stss_update_time'] = date('Y-m-d H:i:s');
            $stss_res = DB::table('stock_sell_storage')->where('stss_id', $now_stss_data->stss_id)->update($stss_data);
        }

        if ($stss_res) {
            return ['code'=>200, 'msg'=>'success', 'data'=>'进销存更新成功'];
        } else {
            return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
        }
    }

    /**
     * [noCostImport 无成本：只计算库存数量，不计算成本金额]
     * @param  [type] $user_info    [用户信息]
     * @param  [type] $shop_id      [总店id]
     * @param  [type] $type         [类型：1采购，2销出，3调拨出库，4调拨入库]
     * @param  [type] $ware_id      [仓库id]
     * @param  [type] $waon_class   [仓库类别]
     * @param  [type] $goo_id       [数组，商品id]
     * @param  [type] $goo_quantity [数组，商品数量；[商品id=>商品数量]]
     * @param  [type] $goo_price    [数组，商品金额，成本价(不乘以100)；[商品id=>商品金额]]
     * @param  [type] $now_year     [当前年份]
     * @param  [type] $now_month    [当前月]
     * @param  [type] $last_year    [上个月的年份]
     * @param  [type] $last_month   [上个月]
     * @return [type]               [description]
     */
    private static function noCostImport(
        $user_info,
        $shop_id,
        $type,
        $ware_id,
        $waon_class,
        $goo_id,
        $goo_quantity,
        $goo_price,
        $now_year,
        $now_month,
        $last_year,
        $last_month
    ) {
        $goo_all_amount = 0;//商品总数量
        foreach ($goo_id as $key => $value) {
            $goo_all_amount += $goo_quantity[$value];
            // 查询商品的当前月份的商品进销存数据
            $now_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $now_year)
                    ->where('sssg_month', $now_month)
                    ->where('sssg_goo_id', $goo_id)
                    ->select('sssg_id', 'sssg_surplus_stock', 'sssg_month_enter_stock', 'sssg_month_out_stock', 'sssg_month_adjust_stock', 'sssg_month_adjust_cost', 'sssg_month_adjust_stock_type')
                    ->first();
            // 进销存商品表
            if (empty($now_sssg_data)) {
                // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
                // 获取上月的期末成本，作为当前月份的期初成本
                $last_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $last_year)
                    ->where('sssg_month', $last_month)
                    ->where('sssg_goo_id', $goo_id)
                    ->select('sssg_id', 'sssg_surplus_stock')
                    ->first();
                $sssg_data['sssg_cid'] = $user_info->adm_cid;//公司id
                $sssg_data['sssg_superior_id'] = $shop_id;//所属总店id
                $sssg_data['sssg_shop_id'] = $user_info->adm_shop_id;// 所属门店id
                $sssg_data['sssg_ware_id'] = $ware_id;//仓库id
                $sssg_data['sssg_waon_class'] = $waon_class;//仓库类别
                $sssg_data['sssg_goo_id'] = $value;//商品id
                $sssg_data['sssg_year'] = $now_year;//年份
                $sssg_data['sssg_month'] = $now_month;//月
                // $sssg_data['sssg_initial_stock'] = isset($last_sssg_data->sssg_surplus_stock) ? $last_sssg_data->sssg_surplus_stock : 0;//期初库存，总库存数量
                $sssg_initial_stock = isset($last_sssg_data->sssg_surplus_stock) ? $last_sssg_data->sssg_surplus_stock : 0;//期初库存，总库存数量
                if ($type == 1) {
                    // 采购
                    $sssg_data['sssg_initial_stock'] = $sssg_initial_stock + $goo_quantity[$value];//期初库存
                    $sssg_data['sssg_surplus_stock'] = $sssg_initial_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')->insert($sssg_data);
            } else {
                if ($type == 1) {
                    // 采购
                    // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                    $sssg_data['sssg_initial_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//期初库存
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                    // $sssg_data['sssg_month_enter_stock'] = $now_sssg_data->sssg_month_enter_stock + $goo_quantity[$value];//月入库存数量
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')
                        ->where('sssg_id', $now_sssg_data->sssg_id)
                        ->update($sssg_data);
            }
            if (!$sssg_res) {
                return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败1'];
            }
        }

        // 进销存查询统计表
        // 查询当前月份、仓库、仓库类别的数据
        $now_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $now_year)
                ->where('stss_month', $now_month)
                ->select('stss_id', 'stss_surplus_stock', 'stss_month_enter_stock', 'stss_month_out_stock', 'stss_month_adjust_stock', 'stss_month_adjust_stock_type')
                ->first();
        if (empty($now_stss_data)) {
            // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
            // 获取上月的期末成本，作为当前月份的期初成本
            $last_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $last_year)
                ->where('stss_month', $last_month)
                ->select('stss_id', 'stss_surplus_stock')
                ->first();
            $stss_data['stss_cid'] = $user_info->adm_cid;//公司id
            $stss_data['stss_superior_id'] = $shop_id;//所属总店id
            $stss_data['stss_shop_id'] = $user_info->adm_shop_id;// 所属门店id
            $stss_data['stss_ware_id'] = $ware_id;//仓库id
            $stss_data['stss_waon_class'] = $waon_class;//仓库类别
            $stss_data['stss_year'] = $now_year;//年份
            $stss_data['stss_month'] = $now_month;//月
            // $stss_data['stss_initial_stock'] = isset($last_stss_data->stss_surplus_stock) ? $last_stss_data->stss_surplus_stock : 0;//期初库存，总库存数量
            $stss_initial_stock = isset($last_stss_data->stss_surplus_stock) ? $last_stss_data->stss_surplus_stock : 0;//期初库存，总库存数量
            if ($type == 1) {
                // 采购
                $stss_data['stss_surplus_stock'] = $stss_initial_stock + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_initial_stock'] = $stss_initial_stock + $goo_all_amount;//期初库存
            }
            $stss_data['stss_update_time'] = date('Y-m-d H:i:s');
            $stss_res = DB::table('stock_sell_storage')->insert($stss_data);
        } else {
            // 当前月份已经添加了商品的进销存，更改商品的进销存数据
            if ($type == 1) {
                // 采购
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock + $goo_all_amount;//结余库存，剩余总库存数量
                $stss_data['stss_initial_stock'] = $now_stss_data->stss_surplus_stock + $goo_all_amount;//期初库存
            }
            $stss_data['stss_update_time'] = date('Y-m-d H:i:s');
            $stss_res = DB::table('stock_sell_storage')->where('stss_id', $now_stss_data->stss_id)->update($stss_data);
        }

        if ($stss_res) {
            return ['code'=>200, 'msg'=>'success', 'data'=>'进销存更新成功'];
        } else {
            return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败2'];
        }
    }

    /**
     * [firstInFirstOut 进销存先进先出算法]
     * @param  [type] $user_info    [用户信息]
     * @param  [type] $shop_id      [总店id]
     * @param  [type] $type         [类型：1采购，2销出，3调拨出库，4调拨入库]
     * @param  [type] $ware_id      [仓库id]
     * @param  [type] $waon_class   [仓库类别]
     * @param  [type] $goo_id       [数组，商品id]
     * @param  [type] $goo_quantity [数组，商品数量；[商品id=>商品数量]]
     * @param  [type] $goo_price    [数组，商品金额，成本价(不乘以100)；[商品id=>商品金额]]
     * @param  [type] $now_year     [当前年份]
     * @param  [type] $now_month    [当前月]
     * @param  [type] $last_year    [上个月的年份]
     * @param  [type] $last_month   [上个月]
     * @return [type]               [description]
     */
    protected static function firstInFirstOutImport(
        $user_info,
        $shop_id,
        $type,
        $ware_id,
        $waon_class,
        $goo_id,
        $goo_quantity,
        $goo_price,
        $now_year,
        $now_month,
        $last_year,
        $last_month
    ) {
        $goo_all_amount = 0;//商品总数量
        $goo_all_price = 0;//商品总金额
        // 先进先出，进销存商品记录表，记录商品数据
        foreach ($goo_id as $key => $value) {
            $now_cost = $goo_quantity[$value] * $goo_price[$value] * 100;
            $goo_all_amount += $goo_quantity[$value];
            if ($type == 1) {
                $goo_all_price += $now_cost;
                // 采购
                // 先进先出，商品记录数据
                $ssgr_data['ssgr_cid'] = $user_info->adm_cid;//公司id
                $ssgr_data['ssgr_superior_id'] = $shop_id;//所属总店id
                $ssgr_data['ssgr_shop_id'] = $user_info->adm_shop_id;// 所属门店id
                $ssgr_data['ssgr_ware_id'] = $ware_id;//仓库id
                $ssgr_data['ssgr_waon_class'] = $waon_class;//仓库类别
                $ssgr_data['ssgr_goo_id'] = $value;//商品id
                $ssgr_data['ssgr_year'] = $now_year;//年份
                $ssgr_data['ssgr_month'] = $now_month;//月
                $ssgr_data['ssgr_initial_stock'] = $goo_quantity[$value];//期初库存，总库存数量
                $ssgr_data['ssgr_initial_average_cost'] = $goo_price[$value] * 100;//期初平均价（单位：分）
                $ssgr_data['ssgr_initial_cost'] = $now_cost;//期初金额-成本（单位：分）,总的成本
                $ssgr_data['ssgr_surplus_stock'] = $goo_quantity[$value];//结余库存，剩余总库存数量
                $ssgr_data['ssgr_surplus_average_cost'] = $goo_price[$value] * 100;//期末平均价（单位：分）
                $ssgr_data['ssgr_surplus_cost'] = $now_cost;//结余成本（单位：分），剩余总的成本
                // $ssgr_data['ssgr_month_enter_stock'] = $goo_quantity[$value];//月入库存数量
                // $ssgr_data['ssgr_month_enter_cost'] = $now_cost;//月入成本（单位：分）
                $ssgr_data['ssgr_update_time'] = date('Y-m-d H:i:s');
                $ssgr_res = DB::table('stock_sell_goods_record')->insert($ssgr_data);
                if (!$ssgr_res) {
                    return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
                }
            }

            // 进销存商品表
            // 查询商品的当前月份的商品进销存数据
            $now_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $now_year)
                    ->where('sssg_month', $now_month)
                    ->where('sssg_goo_id', $value)
                    ->select('sssg_id', 'sssg_surplus_stock', 'sssg_surplus_cost', 'sssg_month_enter_stock', 'sssg_month_enter_cost', 'sssg_month_out_stock', 'sssg_month_out_cost', 'sssg_month_adjust_stock', 'sssg_month_adjust_cost', 'sssg_month_adjust_stock_type', 'sssg_month_adjust_cost_type')
                    ->first();

            if (empty($now_sssg_data)) {
                // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
                // 获取上月的期末成本，作为当前月份的期初成本
                $last_sssg_data = DB::table('stock_sell_storage_goods')
                    ->where('sssg_cid', $user_info->adm_cid)
                    ->where('sssg_shop_id', $user_info->adm_shop_id)
                    ->where('sssg_ware_id', $ware_id)
                    ->where('sssg_waon_class', $waon_class)
                    ->where('sssg_year', $last_year)
                    ->where('sssg_month', $last_month)
                    ->where('sssg_goo_id', $value)
                    ->select('sssg_id', 'sssg_surplus_stock', 'sssg_surplus_cost')
                    ->first();
                $sssg_data['sssg_cid'] = $user_info->adm_cid;//公司id
                $sssg_data['sssg_superior_id'] = $shop_id;//所属总店id
                $sssg_data['sssg_shop_id'] = $user_info->adm_shop_id;// 所属门店id
                $sssg_data['sssg_ware_id'] = $ware_id;//仓库id
                $sssg_data['sssg_waon_class'] = $waon_class;//仓库类别
                $sssg_data['sssg_goo_id'] = $value;//商品id
                $sssg_data['sssg_year'] = $now_year;//年份
                $sssg_data['sssg_month'] = $now_month;//月
                // $sssg_data['sssg_initial_stock'] = isset($last_sssg_data->sssg_surplus_stock) ? $last_sssg_data->sssg_surplus_stock : 0;//期初库存，总库存数量
                // $sssg_data['sssg_initial_cost'] = isset($last_sssg_data->sssg_surplus_cost) ? $last_sssg_data->sssg_surplus_cost : 0;//期初金额-成本（单位：分）,总的成本
                $sssg_initial_stock = isset($last_sssg_data->sssg_surplus_stock) ? $last_sssg_data->sssg_surplus_stock : 0;//期初库存，总库存数量
                $sssg_initial_cost = isset($last_sssg_data->sssg_surplus_cost) ? $last_sssg_data->sssg_surplus_cost : 0;//期初金额-成本（单位：分）,总的成本
                if ($type == 1) {
                    // 采购
                    $sssg_data['sssg_surplus_stock'] = $sssg_initial_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $sssg_initial_cost + $now_cost;//结余成本（单位：分），剩余总的成本
                    $sssg_data['sssg_initial_stock'] = $sssg_initial_stock + $goo_quantity[$value];//期初库存
                    $sssg_data['sssg_initial_cost'] = $sssg_initial_cost + $now_cost;//期初成本
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')->insert($sssg_data);
            } else {
                if ($type == 1) {
                    // 采购
                    // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                    $sssg_data['sssg_surplus_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                    $sssg_data['sssg_surplus_cost'] = $now_sssg_data->sssg_surplus_cost + $now_cost;//结余成本（单位：分），剩余总的成本

                    $sssg_data['sssg_initial_stock'] = $now_sssg_data->sssg_surplus_stock + $goo_quantity[$value];//期初库存
                    $sssg_data['sssg_initial_cost'] = $now_sssg_data->sssg_surplus_cost + $now_cost;//期初成本
                }
                $sssg_data['sssg_update_time'] = date('Y-m-d H:i:s');
                $sssg_res = DB::table('stock_sell_storage_goods')
                        ->where('sssg_id', $now_sssg_data->sssg_id)
                        ->update($sssg_data);
            }

            if (!$sssg_res) {
                return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
            }
        }

        // 进销存查询统计表
        // 查询当前月份、仓库、仓库类别的数据
        $now_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $now_year)
                ->where('stss_month', $now_month)
                ->select('stss_id', 'stss_surplus_stock', 'stss_surplus_cost', 'stss_month_enter_stock', 'stss_month_enter_cost', 'stss_month_out_stock', 'stss_month_out_cost')
                ->first();
        if (empty($now_stss_data)) {
            // 当前月份没有数据，就要添加新的数据，期初库存、期初成本数据
            // 获取上月的期末成本，作为当前月份的期初成本
            $last_stss_data = DB::table('stock_sell_storage')
                ->where('stss_cid', $user_info->adm_cid)
                ->where('stss_shop_id', $user_info->adm_shop_id)
                ->where('stss_ware_id', $ware_id)
                ->where('stss_waon_class', $waon_class)
                ->where('stss_year', $last_year)
                ->where('stss_month', $last_month)
                ->select('stss_id', 'stss_surplus_stock', 'stss_surplus_cost')
                ->first();
            $stss_data['stss_cid'] = $user_info->adm_cid;//公司id
            $stss_data['stss_superior_id'] = $shop_id;//所属总店id
            $stss_data['stss_shop_id'] = $user_info->adm_shop_id;// 所属门店id
            $stss_data['stss_ware_id'] = $ware_id;//仓库id
            $stss_data['stss_waon_class'] = $waon_class;//仓库类别
            $stss_data['stss_year'] = $now_year;//年份
            $stss_data['stss_month'] = $now_month;//月
            // $stss_data['stss_initial_stock'] = isset($last_stss_data->stss_surplus_stock) ? $last_stss_data->stss_surplus_stock : 0;//期初库存，总库存数量
            // $stss_data['stss_initial_cost'] = isset($last_stss_data->stss_surplus_cost) ? $last_stss_data->stss_surplus_cost : 0;//期初金额-成本（单位：分）,总的成本
            $stss_initial_stock = isset($last_stss_data->stss_surplus_stock) ? $last_stss_data->stss_surplus_stock : 0;//期初库存，总库存数量
            $stss_initial_cost = isset($last_stss_data->stss_surplus_cost) ? $last_stss_data->stss_surplus_cost : 0;//期初金额-成本（单位：分）,总的成本
            if ($type == 1) {
                // 采购
                $stss_data['stss_surplus_stock'] = $stss_initial_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $stss_initial_cost + $goo_all_price;//结余成本（单位：分），剩余总的成本

                $stss_data['stss_initial_stock'] = $stss_initial_stock + $goo_quantity[$value];//期初库存
                $stss_data['stss_initial_cost'] = $stss_initial_cost + $goo_all_price;//期初成本
            }
            $stss_data['stss_update_time'] = date('Y-m-d H:i:s');
            $stss_res = DB::table('stock_sell_storage')->insert($stss_data);
        } else {
            if ($type == 1) {
                // 采购
                // 当前月份已经添加了商品的进销存，更改商品的进销存数据
                $stss_data['stss_surplus_stock'] = $now_stss_data->stss_surplus_stock + $goo_quantity[$value];//结余库存，剩余总库存数量
                $stss_data['stss_surplus_cost'] = $now_stss_data->stss_surplus_cost + $goo_all_price;//结余成本（单位：分），剩余总的成本
                $stss_data['stss_initial_stock'] = $now_stss_data->stss_surplus_stock + $goo_quantity[$value];//期初库存
                $stss_data['stss_initial_cost'] = $now_stss_data->stss_surplus_cost + $goo_all_price;//期初成本
            }
            $stss_res = DB::table('stock_sell_storage')->where('stss_id', $now_stss_data->stss_id)->update($stss_data);
        }

        if ($stss_res) {
            return ['code'=>200, 'msg'=>'success', 'data'=>'进销存更新成功'];
        } else {
            return ['code'=>500, 'msg'=>'error', 'data'=>'进销存更新失败'];
        }
    }

    /**
     * [purchaseRecord 添加进销存记录和商品信息]
     * @param  [type] $user_info    [用户信息]
     * @param  [type] $shop_id      [总店id]
     * @param  [type] $cg_id        [采购id]
     * @param  [type] $cg_number    [采购单号]
     * @param  [type] $rk_id        [入库id]
     * @param  [type] $rk_number    [入库单号]
     * @param  [type] $ware_id      [仓库id]
     * @param  [type] $waon_class   [仓库类别]
     * @param  [type] $cg_amount    [采购金额]
     * @param  [type] $goods_amount [商品总量]
     * @param  [type] $goods_data   [数组，商品数据，['商品id'=>['仓库商品id', '商品出库数量','出库类型']]]
     * @return [type]               [description]
     */
    public static function purchaseRecord(
        $user_info,
        $cg_id,
        $cg_number,
        $rk_id,
        $rk_number,
        $ware_id,
        $waon_class,
        $cg_amount,
        $goods_amount,
        $goods_data
    ) {
        // 判断是否为加盟店账号,shop_superior_id 为0 是总店，不为0，是总店id或加盟店id
        $shop_id = $user_info->shop_superior_id ? $user_info->shop_superior_id : $user_info->adm_shop_id;
        $pure_data['pure_cid'] = $user_info->adm_cid;//公司id
        $pure_data['pure_superior_shop_id'] = $shop_id;//总店id
        $pure_data['pure_shop_id'] = $user_info->adm_shop_id;//门店id
        $pure_data['pure_userid'] = $user_info->adm_id;//操作人id
        $pure_data['pure_cg_id'] = $cg_id;//采购id
        $pure_data['pure_cg_number'] = $cg_number;//采购单号
        $pure_data['pure_rk_id'] = $rk_id;//入库id
        $pure_data['pure_rk_number'] = $rk_number;//入库单号
        $pure_data['pure_ware_id'] = $ware_id;//仓库id
        $pure_data['pure_waon_class'] = $waon_class;//仓库类别
        $pure_data['pure_cg_amount'] = $cg_amount;//采购金额
        $pure_data['pure_goods_amount'] = $goods_amount;//商品总量
        $pure_id = DB::table('purchase_record')->insertGetId($pure_data);

        $n = 0;
        foreach ($goods_data as $goo_id_key => $value) {
            $purg_data[$n]['purg_pure_id'] = $pure_id;//进销存记录id
            $purg_data[$n]['purg_rk_number'] = $rk_number;//入库单号
            $purg_data[$n]['purg_ware_id'] = $ware_id;//仓库id
            $purg_data[$n]['purg_waon_class'] = $waon_class;//仓库类别
            $purg_data[$n]['purg_wago_id'] = $value['wago_id'];//仓库商品id
            $purg_data[$n]['purg_goo_id'] = $goo_id_key;//商品id
            $purg_data[$n]['purg_stock_number'] = $value['stock_number'];//良品库存数量
            $purg_data[$n]['purg_priority_stock_number'] = $value['priority_stock_number'];//优先库存数量
            $purg_data[$n]['purg_surplus_stock'] = $value['surplus_stock'];//结余库存
            $purg_data[$n]['purg_univalence'] = $value['univalence'];//采购单价
            $purg_data[$n]['purg_old_univalence'] = $value['old_univalence'];//分算评价后的商品单价
            $purg_data[$n]['purg_surplus_cost'] = $value['surplus_cost'];//结余成本
            $purg_data[$n]['purg_edit_time'] = date('Y-m-d H:i:s');
            $n++;
        }
        
        $purg_id = DB::table('purchase_record_goods')->insert($purg_data);
        if ($purg_id && $pure_id) {
            return true;
        }else{
            return false;
        }
    }
}
