<?php
/*
|--------------------------------------------------------------------------
| 技师级别管理控制器
|--------------------------------------------------------------------------
|
|  @Date: 2021/8/18
|  @Time: 14:14
|  @version 1.0
|  @copyright BlueCity
*/
namespace App\Http\Controllers\Mounting;

use App\Services\LoginUserService;
use App\Services\TechnicalTitleService;
use Illuminate\Http\Request;
use BlueCity\Core\Controllers\MountingCoreController as Controller;
use BlueCity\Core\Traits\ResourceControllerMethod;
use App\Services\TechnicalLevelService;
use BlueCity\Core\Inc\ErrorInc;

class TechnicalLevelController extends Controller
{
    public $dataService = null;  // ResourceControllerMethod 必备参数
    public $dataFields = ['grle_name', 'grpr_id', 'grle_remark'];  // 保存和修改时的 fields 参数
    protected $validateRules = [
        'rules' => [
            'grle_name' => 'required',
            'grpr_id' => 'required|numeric'
        ],
        'messages' => [

        ],
        'attributes' => [
            'grle_name' => '级别名称',
            'grle_grpr_id' => '所属职称',
            'grle_remark' => '备注',
        ]
    ];

    public function __construct()
    {
        $service = new TechnicalLevelService();
        $this->bootWithService($service);
    }

    use ResourceControllerMethod;

    /**
     * @api {get} /api/mounting/technicalLevel 技师级别列表
     * @apiGroup technical
     * @apiName 技师管理
     * @apiVersion 1.0.0
     *
     * @apiDescription 技师级别管理列表
     *
     * @apiParam {String} keywords      关键字
     * @apiParam {String} page          页码
     * @apiParam {String} page_size     页容量
     *
     * @apiSuccess {Number} error       状态码，0：请求成功
     * @apiSuccess {String} msg         提示信息
     * @apiSuccess {Object} data        返回数据
     *
     * @apiSuccessExample {json} Success-Response:
     * {"code":0,"message":"","data":[]}
     *
     * @apiErrorExample {json} Error-Response:
     * {"code":5001,"message":"接口异常","data":[]}
     *
     */


    /**
     * @api {post} /api/mounting/technicalLevel 技师级别信息新增
     * @apiGroup technical
     * @apiName 技师管理
     * @apiVersion 1.0.0
     *
     * @apiDescription 技师级别信息新增
     *
     * @apiParam {string} grle_name     级别名称
     * @apiParam {int} grpr_id          职称编号
     * @apiParam {string} grle_remark   级别备注信息
     *
     * @apiSuccess {Number} error       状态码，0：请求成功
     * @apiSuccess {String} msg         提示信息
     * @apiSuccess {Object} data        返回数据
     *
     * @apiSuccessExample {json} Success-Response:
     * {"code":0,"message":"新增成功","data":[]}
     *
     * @apiErrorExample {json} Error-Response:
     * {"code":5001,"message":"新增失败","data":[]}
     *
     */
    public function _beforeStore($request) {
        $existTitle = TechnicalTitleService::getByID($request->get('grpr_id'), 0, 1);
        if (!$existTitle) {
            return $this->error('参数错误，职位不存在', ErrorInc::INVALID_PARAM);
        }
        $exist = TechnicalLevelService::nameExists($request->get('grle_name'), $request->get('grpr_id'));
        if ($exist) {
            return $this->error('同名内容已经存在', ErrorInc::DATA_EXIST);
        }

        return false;
    }

    public function _setStore($data) {
        $userInfor = LoginUserService::getUser();
        $data['grle_grpr_id'] = $data['grpr_id'];
        $data['grle_remark'] = $data['grle_remark'] ?? '';
        unset($data['grpr_id']);
        $data['grle_shop_id'] = $userInfor['adm_shop_id'];
        $data['grle_superior_shop_id'] = $userInfor['shop_superior_id'];
        $data['grle_cid'] = $userInfor['adm_cid'];
        $data['grle_state'] = 1;
        $data['grle_delete'] = 1;

        return $data;
    }

    /**
     * @api {put} /api/mounting/technicalLevel/{id} 技师级别信息更新
     * @apiGroup technical
     * @apiName 技师管理
     * @apiVersion 1.0.0
     *
     * @apiDescription 技师级别信息更新
     *
     * @apiParam {int} id               编号
     * @apiParam {string} grle_name     级别名称
     * @apiParam {int} grpr_id          职称编号
     * @apiParam {string} grle_remark   级别备注信息
     *
     * @apiSuccess {Number} error       状态码，0：请求成功
     * @apiSuccess {String} msg         提示信息
     * @apiSuccess {Object} data        返回数据
     *
     * @apiSuccessExample {json} Success-Response:
     * {"code":0,"message":"更新成功","data":[]}
     *
     * @apiErrorExample {json} Error-Response:
     * {"code":5001,"message":"更新失败","data":[]}
     *
     */

    public function _beforeUpdate($request, $id) {
        $existTitle = TechnicalTitleService::getByID($request->get('grpr_id'));
        if (!$existTitle) {
            return $this->error('参数错误，职位不存在', ErrorInc::INVALID_PARAM);
        }
        $exist = TechnicalLevelService::nameExists($request->get('grle_name'), $request->get('grpr_id'), $id);
        if ($exist) {
            return $this->error('同名内容已经存在', ErrorInc::DATA_EXIST);
        }

        return false;
    }


    public function _setUpdate($data) {
        $data['grle_grpr_id'] = $data['grpr_id'];
        $data['grle_remark'] = $data['grle_remark'] ?? '';
        unset($data['grpr_id']);

        return $data;
    }


    /**
     * @api {delete} /api/mounting/technicalLevel/{id} 技师级别删除
     * @apiGroup technical
     * @apiName 技师管理
     * @apiVersion 1.0.0
     *
     * @apiDescription 技师级别删除
     *
     * @apiParam {int} id          编号
     *
     * @apiSuccess {Number} error       状态码，0：请求成功
     * @apiSuccess {String} msg         提示信息
     * @apiSuccess {Object} data        返回数据
     *
     * @apiSuccessExample {json} Success-Response:
     * {"code":0,"message":"删除成功","data":[]}
     *
     * @apiErrorExample {json} Error-Response:
     * {"code":5001,"message":"接口异常","data":[]}
     *
     */

    /**
     * @api {post} /api/mounting/technicalLevel/{id}/setEnable 技师级别开关
     * @apiGroup technical
     * @apiName 技师管理
     * @apiVersion 1.0.0
     *
     * @apiDescription 技师级别开关管理
     *
     * @apiParam {int} id          编号
     * @apiParam {int} enable      状态值【1,2】
     *
     * @apiSuccess {Number} error       状态码，0：请求成功
     * @apiSuccess {String} msg         提示信息
     * @apiSuccess {Object} data        返回数据
     *
     * @apiSuccessExample {json} Success-Response:
     * {"code":0,"message":"","data":[]}
     *
     * @apiErrorExample {json} Error-Response:
     * {"code":5001,"message":"接口异常","data":[]}
     *
     */

    public function setEnable(int $ID, Request $request) {
        $entity = TechnicalLevelService::getByID($ID);
        if (empty($entity)) {
            return $this->error('未找到数据', ErrorInc::DATA_EMPTY);
        }
        $state = $request->get('enable');
        $res = TechnicalLevelService::setEnable($ID, $state);
        if ($res) {
            $this->success('更新成功');
        }
        return $this->error('更新失败', ErrorInc::MYSQL_FAILED);
    }
}
