<?php
/*
|--------------------------------------------------------------------------
| 技师、工单绩效项、绩效考核方法控制器
|--------------------------------------------------------------------------
|
|  @Date: 2021/8/18
|  @Time: 14:11
|  @version 1.0
|  @copyright BlueCity
*/

namespace App\Http\Controllers\Mounting;

use App\Services\LoginUserService;
use App\Services\MeritPayOptionsService;
use App\Services\TechnicalLevelService;
use App\Services\TechnicalTitleService;
use Illuminate\Http\Request;
use BlueCity\Core\Inc\ErrorInc;
use BlueCity\Core\Traits\ResourceControllerMethod;
use BlueCity\Core\Controllers\MountingCoreController as Controller;

class MeritPayOptionsController extends Controller
{
    public $dataService = null;  // ResourceControllerMethod 必备参数
    public $dataFields = ['merit_name', 'merit_value', 'title_id', 'level_id', 'm_type'];  // 保存和修改时的 fields 参数
    protected $validateRules = [
        'rules' => [
            'merit_name' => 'required',
            'merit_value' => 'numeric',
            'title_id' => 'numeric',
            'level_id' => 'numeric',
            'm_type' => 'digits:1'
        ],
        'messages' => [

        ],
        'attributes' => [
            'merit_name' => '绩效办法名称',
            'merit_value' => '绩效值',
            'memo' => '备注',
            'title_id' => '职称',
            'level_id' => '级别',
            'm_type' => '绩效类型',
        ]
    ];

    public function __construct()
    {
        $service = new MeritPayOptionsService();
        $this->bootWithService($service);
    }

    use ResourceControllerMethod;

    /**
     * @api {get} /api/mounting/meritPayOptions 绩效项列表
     * @apiGroup meritPayOptions
     * @apiName 绩效项管理
     * @apiVersion 1.0.0
     *
     * @apiDescription 绩效项管理列表
     *
     * @apiParam {String} keywords      关键字
     * @apiParam {int} page          页码
     * @apiParam {int} page_size     页容量
     *
     * @apiSuccess {int} error       状态码，0：请求成功
     * @apiSuccess {String} msg         提示信息
     * @apiSuccess {Object} data        返回数据
     *
     * @apiSuccessExample {json} Success-Response:
     * {"code":0,"message":"","data":[]}
     *
     * @apiErrorExample {json} Error-Response:
     * {"code":5001,"message":"接口异常","data":[]}
     *
     */



    /**
     * @api {post} /api/mounting/meritPayOptions 绩效项增加
     * @apiGroup meritPayOptions
     * @apiName 绩效项管理
     * @apiVersion 1.0.0
     *
     * @apiDescription 绩效项增加
     *
     * @apiParam {int} m_type           类型
     * @apiParam {String} merit_name    绩效名称
     * @apiParam {int} parts_id         车部位ID
     * @apiParam {int} value_type       绩效值类型
     * @apiParam {int} merit_value      绩效值
     * @apiParam {int} title_id         职称ID
     * @apiParam {int} level_id         级别ID
     * @apiParam {String} memo          备注
     *
     * @apiSuccess {Number} error       状态码，0：请求成功
     * @apiSuccess {String} msg         提示信息
     * @apiSuccess {Object} data        返回数据
     *
     * @apiSuccessExample {json} Success-Response:
     * {"code":0,"message":"","data":[]}
     *
     * @apiErrorExample {json} Error-Response:
     * {"code":5001,"message":"接口异常","data":[]}
     *
     */

    public function _beforeStore(Request $request) {
        return $this->beforeCheck($request);
    }

    public function _setStore($request) {
        $data = $this->setData($request);
        $loginUser = LoginUserService::getUser();

        $data['com_id'] = $loginUser['adm_cid'];
        $data['superior_shop_id'] = $loginUser['shop_superior_id'];
        $data['shop_id'] = $loginUser['adm_shop_id'];
        $data['create_uid'] = $loginUser['adm_id'];

        return $data;
    }

    /**
     * @api {put} /api/mounting/meritPayOptions/{ID} 绩效项修改
     * @apiGroup meritPayOptions
     * @apiName 绩效项管理
     * @apiVersion 1.0.0
     *
     * @apiDescription 绩效项修改
     *
     * @apiParam {int} m_type           类型
     * @apiParam {String} merit_name    绩效名称
     * @apiParam {int} parts_id         车部位ID
     * @apiParam {int} value_type       绩效值类型
     * @apiParam {int} merit_value      绩效值
     * @apiParam {int} title_id         职称ID
     * @apiParam {int} level_id         级别ID
     * @apiParam {String} memo          备注
     *
     * @apiSuccess {Number} error       状态码，0：请求成功
     * @apiSuccess {String} msg         提示信息
     * @apiSuccess {Object} data        返回数据
     *
     * @apiSuccessExample {json} Success-Response:
     * {"code":0,"message":"","data":[]}
     *
     * @apiErrorExample {json} Error-Response:
     * {"code":5001,"message":"接口异常","data":[]}
     *
     */


    public function _beforeUpdate(Request $request, $id) {
        return $this->beforeCheck($request);
    }

    public function _setUpdate($request) {
        return $this->setData($request);
    }

    public function setData($request) {
        $loginUser = LoginUserService::getUser();
        $type = $request['m_type'];
        $meritName = $request['merit_name'];
        $partsID = 0;
        if (1 == $type) {
            $valueType = 1;
            $partsID = $request['parts_id'] ?? 0;
        } elseif (2 == $type) {
            $valueType = 1;
        } else {
            $valueType = $request['value_type'] ?? 1;
        }
        $titleID = $request['title_id'] ?? 0;
        $levelID = $request['level_id'] ?? 0;
        $meritMemo = $request['memo'] ?? '';
        $meritValue = $request['merit_value'] ?? 0;
        if ($valueType == 2) {
            $meritValue = $meritValue*100;
        }
        $data = [
            'merit_type' => $type,
            'merit_name' => $meritName,
            'parts_id' => $partsID,
            'tech_title_id' => $titleID,
            'tech_level_id' => $levelID,
            'value_type' => $valueType,
            'merit_value' => $meritValue,
            'memo' => $meritMemo,
            'last_uid' => $loginUser['adm_id'],
            'enabled' => 1,
        ];

        return $data;
    }

    public function beforeCheck(Request $request) {
        $meritName = $request->get('merit_name');
        if (strlen($meritName) > 30) {
            return $this->error('绩效方法名称过长', ErrorInc::INVALID_PARAM);
        }
        $titleID = $request->get('title_id') ?? 0;
        if ($titleID > 0) {
            $title = TechnicalTitleService::getByID($titleID);
            if (empty($title)) {
                return $this->error('职称选择错误');
            }
        }
        $levelID = $request->get('level_id') ?? 0;
        if ($levelID > 0) {
            $level = TechnicalLevelService::getByID($levelID);
            if (empty($level)) {
                return $this->error('职称级别选择错误');
            }
        }

        return false;
    }

    /**
     * @api {post} /api/mounting/meritPayOptions/{id}/setEnable 绩效项开关
     * @apiGroup meritPayOptions
     * @apiName 绩效项管理
     * @apiVersion 1.0.0
     *
     * @apiDescription 绩效项开关
     *
     * @apiParam {int} id          编号
     * @apiParam {int} enable      状态值【1,2】
     *
     * @apiSuccess {Number} error       状态码，0：请求成功
     * @apiSuccess {String} msg         提示信息
     * @apiSuccess {Object} data        返回数据
     *
     * @apiSuccessExample {json} Success-Response:
     * {"code":0,"message":"","data":[]}
     *
     * @apiErrorExample {json} Error-Response:
     * {"code":5001,"message":"接口异常","data":[]}
     *
     */

    public function setEnable(int $ID, Request $request)
    {
        $entity = MeritPayOptionsService::getByID($ID);
        if (empty($entity)) {
            return $this->error('未找到数据', ErrorInc::DATA_EMPTY);
        }
        $state = $request->get('enable');
        if ($state != 1 && $state != 2) {
            return $this->error('状态码错误', ErrorInc::INVALID_PARAM);
        }
        $res = MeritPayOptionsService::setEnable($ID, $state);
        if ($res) {
            $this->success('更新成功');
        }
        return $this->error('更新失败', ErrorInc::MYSQL_FAILED);
    }

    /**
     * @api {delete} /api/mounting/meritPayOptions/{id} 绩效项删除
     * @apiGroup meritPayOptions
     * @apiName 绩效项管理
     * @apiVersion 1.0.0
     *
     * @apiDescription 绩效项删除
     *
     * @apiParam {int} id          编号
     *
     * @apiSuccess {Number} error       状态码，0：请求成功
     * @apiSuccess {String} msg         提示信息
     * @apiSuccess {Object} data        返回数据
     *
     * @apiSuccessExample {json} Success-Response:
     * {"code":0,"message":"","data":[]}
     *
     * @apiErrorExample {json} Error-Response:
     * {"code":5001,"message":"接口异常","data":[]}
     *
     */
    public function destroy(int $id)
    {
        $entity = $this->dataService->find($id);
        if ($entity) {
            $loginUser = LoginUserService::getUser();
            if($entity['shop_id'] != $loginUser['adm_shop_id']) {
                return $this->error('该内容不存在', ErrorInc::DATA_EMPTY);
            }
            $this->dataService->delete($id);
            if (method_exists($this, '_afterDestroy')) {
                $this->_afterDestroy($entity);
            }
            return $this->success('删除成功');
        } else {
            return $this->error('该内容不存在', ErrorInc::DATA_EMPTY);
        }
    }
}
