<?php

namespace App\Http\Controllers\Mounting\Business;

use Illuminate\Http\Request;
use BlueCity\Core\Controllers\MountingCoreController as Controller;
use BlueCity\Core\Traits\ResourceControllerMethod;
use App\Services\Business\WashAfterSalesService;
use BlueCity\Core\Inc\ErrorInc;
use App\Services\LoginUserService;
use App\Models\OperationLog;
use Illuminate\Support\Facades\DB;
use App\Services\CommonService;

class WashAfterSalesController extends Controller
{
    public $dataService = null;

    public function __construct()
    {
        $this->dataService = $service = new WashAfterSalesService();
        $this->bootWithService($service);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $user_info = LoginUserService::getUser();
        $request['show_count'] = isset($request['show_count']) ?$request['show_count']:10;
        $search_data = $this->dataService->searchPage($user_info, $request->all(), $request['show_count']);
        if(isset($request['is_excel']) && $request['is_excel']==2){
            return $search_data;
        }
        // 获取当前列表下的按钮
        $list_button = getListButton($request['api_token'], $request['pri_id']);
        //获取top按钮
        $top_button_data = 'export';
        $top_button = CommonService::getTopButton($top_button_data,'',$list_button);
        if (empty($search_data)) {
            return $this->error('未找到数据', ErrorInc::DATA_EMPTY);
        } else {
            return response()->json(['code'=>200, 'msg'=>'success', 'data'=>$search_data, 'list_button'=>$list_button, 'top_button'=>$top_button]);
        }
    }

    // 详情
    public function info(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->infoValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();
        $return_data = WashAfterSalesService::info($user_info, $request->all());
        return response()->json($return_data);
    }

    // 确认处理
    public function dealWith(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->dealWithValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();
        if ($request['type'] == 1) {
            $return_data = WashAfterSalesService::info($user_info,$request);
            return response()->json($return_data);
        }
        DB::beginTransaction();
        // 确认处理
        $return_data = WashAfterSalesService::dealWith($user_info, $request->all());
        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        }

        // 生成操作记录
        $orlo_res = orderLogType($user_info, $request['afsl_id'], 2, '售后处理');
        if ($request['remark']) {
            // 添加备注
            $orre_res = orderRemarkType($user_info, $request['afsl_id'], 2, $request['remark'], '售后单列表-售后处理');
        } else {
            $orre_res = true;
        }
        if ($orlo_res && $orre_res) {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '售后单列表', '售后处理', '售后处理' . $return_data['data']['afsl_after_number'] . '、订单号：' . $return_data['data']['afsl_order_number'] . '成功');
        } else {
            DB::rollBack();
        }
        return response()->json($return_data);
    }

    // 撤销售后
    public function cancel(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->dealWithValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();

        DB::beginTransaction();
        // 撤销售后
        $return_data = WashAfterSalesService::cancel($user_info, $request->all());
        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        }

        // 生成操作记录
        $orlo_res = orderLogType($user_info, $request['afsl_id'], 2, '撤销售后');
        if ($request['remark']) {
            // 添加备注
            $orre_res = orderRemarkType($user_info, $request['afsl_id'], 2, $request['remark'], '售后单列表-撤销售后');
        } else {
            $orre_res = true;
        }
        if ($orlo_res && $orre_res) {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '售后单列表', '撤销售后', '撤销售后' . $return_data['data']['afsl_after_number'] . '、订单号：' . $return_data['data']['afsl_order_number'] . '成功');
            return $this->success('撤销售后成功');
        } else {
            DB::rollBack();
            return $this->error('撤销售后失败', ErrorInc::MYSQL_FAILED);
        }
    }

    /**
     * 指派技师
     * @authors ldj
     * @email  909490343@qq.com
     * @date   2024-07-10
     * @param  Request          $request [description]
     * @return [type]                    [description]
     */
    public function assignTechnician(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->assignTechnicianValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();

        DB::beginTransaction();
        // 申请完成
        $return_data = WashAfterSalesService::assignTechnician($user_info, $request->all());
        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        }
        $new_data = DB::table('wash_after_sale_list')
            ->where('afsl_id', $request['afsl_id'])
            ->first();
        // 生成操作记录
        $orlo_res = orderLogType($user_info, $request['afsl_id'], 2, '指派技师');
        if ($request['remark']) {
            // 添加备注
            $orre_res = orderRemarkType($user_info, $request['afsl_id'], 2, $request['remark'], '售后单列表-指派技师');
        } else {
            $orre_res = true;
        }
        if ($orlo_res && $orre_res) {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '售后单列表', '指派技师', '指派技师' . $new_data->afsl_after_number . '、订单号：' . $new_data->afsl_order_number . '成功');
            return $this->success('指派技师成功');
        } else {
            DB::rollBack();
            return $this->error('申请完成失败', ErrorInc::MYSQL_FAILED);
        }
    }

    // 申请完成
    public function applyComplete(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->dealWithValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();

        DB::beginTransaction();
        // 申请完成
        $return_data = WashAfterSalesService::applyComplete($user_info, $request->all());
        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        }

        // 生成操作记录
        $orlo_res = orderLogType($user_info, $request['afsl_id'], 2, '申请完成');
        if ($request['remark']) {
            // 添加备注
            $orre_res = orderRemarkType($user_info, $request['afsl_id'], 2, $request['remark'], '售后单列表-申请完成');
        } else {
            $orre_res = true;
        }
        if ($orlo_res && $orre_res) {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '售后单列表', '申请完成', '申请完成' . $return_data['data']['afsl_after_number'] . '、订单号：' . $return_data['data']['afsl_order_number'] . '成功');
            return $this->success('申请完成成功');
        } else {
            DB::rollBack();
            return $this->error('申请完成失败', ErrorInc::MYSQL_FAILED);
        }
    }

    /**
     * 审核驳回技师售后记录
     * @authors ldj
     * @email  909490343@qq.com
     * @date   2024-07-10
     * @param  Request          $request [description]
     * @return [type]                    [description]
     */
    public function audit(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->auditValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();

        DB::beginTransaction();
        // 审核通过
        $return_data = WashAfterSalesService::audit($user_info, $request->all());
        if ($return_data['code'] != 200) {
            DB::rollBack();
        } else {
            DB::commit();
        }
        return response()->json($return_data);
    }

    // 审核通过
    public function auditPass(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->dealWithValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();

        DB::beginTransaction();
        // 审核通过
        $return_data = WashAfterSalesService::auditPass($user_info, $request->all());
        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        }

        // 生成操作记录
        $orlo_res = orderLogType($user_info, $request['afsl_id'], 2, '审核通过');
        if ($request['remark']) {
            // 添加备注
            $orre_res = orderRemarkType($user_info, $request['afsl_id'], 2, $request['remark'], '售后单列表-审核通过');
        } else {
            $orre_res = true;
        }
        if ($orlo_res && $orre_res) {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '售后单列表', '审核通过', '审核通过' . $return_data['data']['afsl_after_number'] . '、订单号：' . $return_data['data']['afsl_order_number'] . '成功');
            return $this->success('审核通过成功');
        } else {
            DB::rollBack();
            return $this->error('审核通过失败', ErrorInc::MYSQL_FAILED);
        }
    }

    // 审核驳回
    public function auditReject(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->auditRejectValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();

        DB::beginTransaction();
        // 审核驳回
        $return_data = WashAfterSalesService::auditReject($user_info, $request->all());
        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        }
        if ($return_data['code'] != 200) {
            DB::rollBack();
        } else {
            DB::commit();
        }
        return response()->json($return_data);
    }

    // 备注
    public function remark(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->remarkValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();
        $return_data = WashAfterSalesService::remarkSearch($user_info, $request->all());
        $orre_res = orderRemarkType($user_info, $request['afsl_id'], 2, $request['remark'], '售后单列表');
        if ($orre_res) {
            OperationLog::addLog($user_info, getRealIp($request), '售后单列表', '备注', '售后单 ' . $return_data['afsl_after_number'] . '、订单号：' . $return_data['afsl_order_number'] . ' 添加备注 :' . $request['remark']);
            return $this->success('备注成功');
        } else {
            return $this->error('备注失败', ErrorInc::MYSQL_FAILED);
        }
    }

    /**
     * 新增售后标签
     * @authors ldj
     * @email  909490343@qq.com
     * @date   2024-07-08
     * @param  Request          $request [description]
     */
    public function addLab(Request $request) {
        // 验证参数
        $error_message = $this->dataService->mainValidator->addLabValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();
        $return_data = WashAfterSalesService::addLab($user_info, $request->all());
        if ($return_data['code'] == 200) {
            OperationLog::addLog($user_info, getRealIp($request), '售后单列表', '备注', '售后单 ' . $return_data['afsl_after_number'] . '、订单号：' . $return_data['afsl_order_number'] . ' 添加备注 :' . $request['remark']);
            return $this->success('备注成功');
        } else {
            return $this->error('备注失败', ErrorInc::MYSQL_FAILED);
        }
    }
}
