<?php

namespace App\Http\Controllers\Mounting\BasicManage;

use Illuminate\Http\Request;
use BlueCity\Core\Controllers\MountingCoreController as Controller;
use BlueCity\Core\Traits\ResourceControllerMethod;
use App\Services\BasicManagement\ProductionManageService;
use App\Services\BasicManagement\ProductionContentService;
use BlueCity\Core\Inc\ErrorInc;
use App\Services\LoginUserService;
use App\Models\OperationLog;
use Illuminate\Support\Facades\DB;

// 基础管理-作品管理
class ProductionManageController extends Controller
{
    public $dataService = null;

    public function __construct()
    {
        $this->dataService = $service = new ProductionManageService();
        $this->contentService = new ProductionContentService();
        $this->bootWithService($service);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->indexValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        $user_info = LoginUserService::getUser();
        if (in_array($request['type'], [1, 2])) {
            // 作品管理数据表，作品分类，作品模块
            $search_data = $this->dataService->searchPage($user_info, $request->all(), $request['show_count']);
        } else {
            // 类型：1作品分类2作品模块3视频4图文
            if ($request['type'] == '3') {
                $request['type'] = '1'; //类型：1视频2图文
            } else if ($request['type'] == '4') {
                $request['type'] = '2';
            }
            // 作品内容数据表，视频，图文
            $search_data = $this->contentService->searchPage($user_info, $request->all(), $request['show_count']);
        }

        // 获取当前列表下的按钮
        $list_button = getListButton($request['api_token'], $request['pri_id']);
        if (empty($search_data)) {
            return $this->error('未找到数据', ErrorInc::DATA_EMPTY);
        } else {
            return response()->json(['code'=>200, 'msg'=>'success', 'data'=>$search_data, 'list_button'=>$list_button]);
        }
    }

    // 详情
    public function info(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->infoValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }
        // 类型：1作品分类2作品模块3视频4图文
        if ($request['type'] == '3') {
            $request['type'] = '1'; //类型：1视频2图文
        } else if ($request['type'] == '4') {
            $request['type'] = '2';
        }
        // 获取用户信息
        $user_info = LoginUserService::getUser();
        $return_data = ProductionContentService::info($user_info, $request->all());
        return response()->json($return_data);
    }

    // 新增
    public function add(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->addValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();

        DB::beginTransaction();
        if (in_array($request['type'], [1, 2])) {
            // 作品分类，作品模块新增
            // 判断名称是否存在
            $exist = ProductionManageService::nameExists($user_info, $request->all(), 0);
            if ($exist) {
                return $this->error('同名内容已经存在', ErrorInc::DATA_EXIST);
            }
            // 新增
            $return_data = ProductionManageService::add($user_info, $request->all());
        } else {
            // 类型：1作品分类2作品模块3视频4图文
            if ($request['type'] == '3') {
                $request['type'] = '1'; //类型：1视频2图文
            } else if ($request['type'] == '4') {
                $request['type'] = '2';
            }
            // 视频、图文新增
            // 判断名称是否存在
            $exist = ProductionContentService::nameExists($user_info, $request->all(), 0);
            if ($exist) {
                return $this->error('同名内容已经存在', ErrorInc::DATA_EXIST);
            }
            // 新增
            $return_data = ProductionContentService::add($user_info, $request->all());
        }
        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        } else {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '作品管理', '新增', '新增' . $request['name'] . '成功');
            return $this->success('新增成功');
        }
    }

    // 编辑
    public function edit(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->editValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();

        DB::beginTransaction();
        if (in_array($request['type'], [1, 2])) {
            // 作品分类，作品模块编辑
            // 判断名称是否存在
            $exist = ProductionContentService::nameExists($user_info, $request->all(), $request['id']);
            if ($exist) {
                return $this->error('同名内容已经存在', ErrorInc::DATA_EXIST);
            }
            // 编辑
            $return_data = ProductionManageService::edit($user_info, $request->all());
        } else {
            // 类型：1作品分类2作品模块3视频4图文
            if ($request['type'] == '3') {
                $request['type'] = '1'; //类型：1视频2图文
            } else if ($request['type'] == '4') {
                $request['type'] = '2';
            }
            // 视频、图文编辑
            // 判断名称是否存在
            $exist = ProductionContentService::nameExists($user_info, $request->all(), $request['id']);
            if ($exist) {
                return $this->error('同名内容已经存在', ErrorInc::DATA_EXIST);
            }
            // 编辑
            $return_data = ProductionContentService::edit($user_info, $request->all());
        }
        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        } else {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '作品管理', '编辑', '编辑' . $request['name'] . '成功');
            return $this->success('编辑成功');
        }
    }

    // 启用
    public function start(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->startValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();
        DB::beginTransaction();
        if (in_array($request['type'], [1, 2])) {
            // 作品分类，作品模块启用
            $return_data = ProductionManageService::start($user_info, $request->all());
            // 获取启用的名称
            $field_columns = ['prma_name'];
            $find_data = ProductionManageService::getFieldsByIdAll($user_info, $request->all(), $field_columns);
            $find_name = array_column($find_data, 'prma_name');
        } else {
            // 类型：1作品分类2作品模块3视频4图文
            if ($request['type'] == '3') {
                $request['type'] = '1'; //类型：1视频2图文
            } else if ($request['type'] == '4') {
                $request['type'] = '2';
            }
            $return_data = ProductionContentService::start($user_info, $request->all());
            // 获取启用的名称
            $field_columns = ['prco_name'];
            $find_data = ProductionContentService::getFieldsByIdAll($user_info, $request->all(), $field_columns);
            $find_name = array_column($find_data, 'prco_name');
        }

        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        } else {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '作品管理', '启用', '启用' . implode(',', $find_name) . '成功');
            return $this->success('启用成功');
        }
    }

    // 停用
    public function stop(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->startValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();
        DB::beginTransaction();
        if (in_array($request['type'], [1, 2])) {
            // 作品分类，作品模块停用
            $return_data = ProductionManageService::stop($user_info, $request->all());
            // 获取停用的名称
            $field_columns = ['prma_name'];
            $find_data = ProductionManageService::getFieldsByIdAll($user_info, $request->all(), $field_columns);
            $find_name = array_column($find_data, 'prma_name');
        } else {
            // 类型：1作品分类2作品模块3视频4图文
            if ($request['type'] == '3') {
                $request['type'] = '1'; //类型：1视频2图文
            } else if ($request['type'] == '4') {
                $request['type'] = '2';
            }
            $return_data = ProductionContentService::stop($user_info, $request->all());
            // 获取停用的名称
            $field_columns = ['prco_name'];
            $find_data = ProductionContentService::getFieldsByIdAll($user_info, $request->all(), $field_columns);
            $find_name = array_column($find_data, 'prco_name');
        }

        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        } else {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '作品管理', '停用', '停用' . implode(',', $find_name) . '成功');
            return $this->success('停用成功');
        }
    }

    // 删除
    public function delete(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->startValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();
        DB::beginTransaction();
        if (in_array($request['type'], [1, 2])) {
            // 作品分类，作品模块删除
            $return_data = ProductionManageService::remove($user_info, $request->all());
            // 获取删除的名称
            $field_columns = ['prma_name'];
            $find_data = ProductionManageService::getFieldsByIdAll($user_info, $request->all(), $field_columns);
            $find_name = array_column($find_data, 'prma_name');
        } else {
            // 类型：1作品分类2作品模块3视频4图文
            if ($request['type'] == '3') {
                $request['type'] = '1'; //类型：1视频2图文
            } else if ($request['type'] == '4') {
                $request['type'] = '2';
            }
            $return_data = ProductionContentService::remove($user_info, $request->all());
            // 获取删除的名称
            $field_columns = ['prco_name'];
            $find_data = ProductionContentService::getFieldsByIdAll($user_info, $request->all(), $field_columns);
            $find_name = array_column($find_data, 'prco_name');
        }

        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        } else {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '作品管理', '删除', '删除' . implode(',', $find_name) . '成功');
            return $this->success('删除成功');
        }
    }
}
