<?php

namespace App\Http\Controllers\Mounting\BasicManage;

use Illuminate\Http\Request;
use BlueCity\Core\Controllers\MountingCoreController as Controller;
use BlueCity\Core\Traits\ResourceControllerMethod;
use App\Services\BasicManagement\ValidateCarManageService;
use BlueCity\Core\Inc\ErrorInc;
use App\Services\LoginUserService;
use App\Models\OperationLog;

class ValidateCarManageController extends Controller
{
    public function __construct()
    {
        $this->dataService = $service = new ValidateCarManageService();
        $this->bootWithService($service);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->indexValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 查询字段
        $field_columns = ['vacm_id', 'vacm_unusual_name', 'vacm_type', 'vacm_remark', 'vacm_state', 'vacm_create_time'];
        $user_info = LoginUserService::getUser();

        $search_data = $this->dataService->searchPage($user_info, $request->all(), $request['show_count'], $field_columns);

        if (empty($search_data)) {
            return $this->error('未找到数据', ErrorInc::DATA_EMPTY);
        } else {
            // 获取当前列表下的按钮
            $list_button = getListButton($request['api_token'], $request['pri_id']);
            return response()->json(['code'=>200, 'msg'=>'success', 'data'=>$search_data, 'list_button'=>$list_button]);
        }
    }

    // 新增
    public function add(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->addValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();
        
        $exist = ValidateCarManageService::nameExists($user_info, $request['vacm_unusual_name'], $request['vacm_type'], 0);
        if ($exist) {
            return $this->error('同名内容已经存在', ErrorInc::DATA_EXIST);
        }
        $res = ValidateCarManageService::add($user_info, $request->all());
        if ($res) {
            $hint = $this->typeHint($request['vacm_type']);
            OperationLog::addLog($user_info, getRealIp($request), '验车管理', '新增', '新增'.$hint.':'.$request['vacm_unusual_name'].'成功');
            return $this->success('新增成功');
        } else {
            return $this->error('新增失败', ErrorInc::MYSQL_FAILED);
        }
    }

    // 编辑
    public function edit(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->editValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }
        
        // 获取用户信息
        $user_info = LoginUserService::getUser();
        $exist = ValidateCarManageService::nameExists($user_info, $request['vacm_unusual_name'], $request['vacm_type'], $request['vacm_id']);
        if ($exist) {
            return $this->error('同名内容已经存在', ErrorInc::DATA_EXIST);
        }

        $res = ValidateCarManageService::edit($request->all());
        if ($res) {
            $hint = $this->typeHint($request['vacm_type']);
            OperationLog::addLog($user_info, getRealIp($request), '验车管理', '编辑', '编辑'.$hint.':'.$request['vacm_unusual_name'].'成功');
            return $this->success('编辑成功');
        } else {
            return $this->error('编辑失败', ErrorInc::MYSQL_FAILED);
        }
    }

    // 启用
    public function start(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->startValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }
        
        // 获取用户信息
        $user_info = LoginUserService::getUser();

        $res = ValidateCarManageService::editState($request['vacm_id'], 1);
        if ($res) {
            // 获取启用的名称
            $field_columns = ['vacm_unusual_name'];
            $find_vacm = ValidateCarManageService::getFieldsByIdAll($user_info, $request['vacm_id'], $field_columns);
            $vacm_unusual_name = array_column($find_vacm, 'vacm_unusual_name');

            $hint = $this->typeHint($request['vacm_type']);
            OperationLog::addLog($user_info, getRealIp($request), '验车管理', '启用', $hint.'：（'.implode(',', $vacm_unusual_name).'）启用成功');
            return $this->success('启用成功');
        } else {
            return $this->error('启用失败', ErrorInc::MYSQL_FAILED);
        }
    }

    // 停用
    public function stop(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->startValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }
        
        // 获取用户信息
        $user_info = LoginUserService::getUser();

        $res = ValidateCarManageService::editState($request['vacm_id'], 2);
        if ($res) {
            // 获取停用的名称
            $field_columns = ['vacm_unusual_name'];
            $find_vacm = ValidateCarManageService::getFieldsByIdAll($user_info, $request['vacm_id'], $field_columns);
            $vacm_unusual_name = array_column($find_vacm, 'vacm_unusual_name');

            $hint = $this->typeHint($request['vacm_type']);
            OperationLog::addLog($user_info, getRealIp($request), '验车管理', '停用', $hint.'：（'.implode(',', $vacm_unusual_name).'）停用成功');
            return $this->success('停用成功');
        } else {
            return $this->error('停用失败', ErrorInc::MYSQL_FAILED);
        }
    }
    
    // 删除
    public function delete(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->startValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }
        
        // 获取用户信息
        $user_info = LoginUserService::getUser();

        $res = ValidateCarManageService::editDelete($request['vacm_id'], 2);
        if ($res) {
            // 获取删除的名称
            $field_columns = ['vacm_unusual_name'];
            $find_vacm = ValidateCarManageService::getFieldsByIdAll($user_info, $request['vacm_id'], $field_columns);
            $vacm_unusual_name = array_column($find_vacm, 'vacm_unusual_name');

            $hint = $this->typeHint($request['vacm_type']);
            OperationLog::addLog($user_info, getRealIp($request), '验车管理', '删除', $hint.'：（'.implode(',', $vacm_unusual_name).'）删除成功');
            return $this->success('删除成功');
        } else {
            return $this->error('删除失败', ErrorInc::MYSQL_FAILED);
        }
    }

    // 搜索验车管理数据
    public function searchValidateCar(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->searchValidateCarValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 查询字段
        $field_columns = ['vacm_id', 'vacm_unusual_name'];
        $user_info = LoginUserService::getUser();

        $search_data = $this->dataService->searchPage($user_info, $request->all(), 10, $field_columns);
        
        if (empty($search_data)) {
            return $this->error('未找到数据', ErrorInc::DATA_EMPTY);
        } else {
            return $this->success('查询成功', $search_data);
        }
    }

    // 类型提示信息
    private function typeHint($type)
    {
        if ($type == 1) {
            $hint = '车内异常管理';
        } else {
            $hint = '外观异常管理';
        }
        return $hint;  
    }
}
