<?php

namespace App\Http\Controllers\Mounting\BasicManage;

use Illuminate\Http\Request;
use BlueCity\Core\Controllers\MountingCoreController as Controller;
use BlueCity\Core\Traits\ResourceControllerMethod;
use App\Services\BasicManagement\GradeProfessionalService;
use App\Services\BasicManagement\GradeLevelService;
use BlueCity\Core\Inc\ErrorInc;
use App\Services\LoginUserService;
use App\Models\OperationLog;
use Illuminate\Support\Facades\DB;

// 基础管理-职级管理
class GradeProfessionalController extends Controller
{
    public $dataService = null;

    public function __construct()
    {
        $this->dataService = $service = new GradeProfessionalService();
        $this->contentService = new GradeLevelService();
        $this->bootWithService($service);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->indexValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }
        
        $user_info = LoginUserService::getUser();
        if (in_array($request['type'], [1])) {
            // 职称管理列表
            $search_data = $this->dataService->searchPage($user_info, $request->all(), $request['show_count']);
        } else {
            // 级别管理列表
            $search_data = $this->contentService->searchPage($user_info, $request->all(), $request['show_count']);
        }

        if (empty($search_data)) {
            return $this->error('未找到数据', ErrorInc::DATA_EMPTY);
        } else {
            // 获取当前列表下的按钮
            $list_button = getListButton($request['api_token'], $request['pri_id']);
            return response()->json(['code'=>200, 'msg'=>'success', 'data'=>$search_data, 'list_button'=>$list_button]);
        }
    }

    // 详情
    public function info(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->infoValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();
        $return_data = GradeLevelService::info($user_info, $request->all());
        return response()->json($return_data);
    }

    // 新增
    public function add(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->addValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();

        DB::beginTransaction();
        if (in_array($request['type'], [1])) {
            // 职称管理
            // 判断名称是否存在
            $exist = GradeProfessionalService::nameExists($user_info, $request->all(), 0);
            if ($exist) {
                return $this->error('同名内容已经存在', ErrorInc::DATA_EXIST);
            }
            // 新增
            $return_data = GradeProfessionalService::add($user_info, $request->all());
        } else {
            // 级别管理
            // 判断名称是否存在
            $exist = GradeLevelService::nameExists($user_info, $request->all(), 0);
            if ($exist) {
                return $this->error('同名内容已经存在', ErrorInc::DATA_EXIST);
            }
            // 新增
            $return_data = GradeLevelService::add($user_info, $request->all());
        }
        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        } else {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '职级管理', '新增', '新增' . $request['name'] . '成功');
            return $this->success('新增成功');
        }
    }

    // 编辑
    public function edit(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->editValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();

        DB::beginTransaction();
        if (in_array($request['type'], [1])) {
            // 职称管理
            // 判断名称是否存在
            $exist = GradeProfessionalService::nameExists($user_info, $request->all(), $request['id']);
            if ($exist) {
                return $this->error('同名内容已经存在', ErrorInc::DATA_EXIST);
            }
            // 编辑
            $return_data = GradeProfessionalService::edit($user_info, $request->all());
        } else {
            // 级别管理
            // 判断名称是否存在
            $exist = GradeLevelService::nameExists($user_info, $request->all(), $request['id']);
            if ($exist) {
                return $this->error('同名内容已经存在', ErrorInc::DATA_EXIST);
            }
            // 编辑
            $return_data = GradeLevelService::edit($user_info, $request->all());
        }
        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        } else {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '职级管理', '编辑', '编辑' . $request['name'] . '成功');
            return $this->success('编辑成功');
        }
    }

    // 启用
    public function start(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->startValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();
        DB::beginTransaction();
        if (in_array($request['type'], [1])) {
            // 职称管理启用
            $return_data = GradeProfessionalService::start($user_info, $request->all());
            // 获取启用的名称
            $field_columns = ['grpr_name'];
            $find_data = GradeProfessionalService::getFieldsByIdAll($user_info, $request->all(), $field_columns);
            $find_name = array_column($find_data, 'grpr_name');
        } else {
            // 级别名称
            $return_data = GradeLevelService::start($user_info, $request->all());
            // 获取启用的名称
            $field_columns = ['grle_name'];
            $find_data = GradeLevelService::getFieldsByIdAll($user_info, $request->all(), $field_columns);
            $find_name = array_column($find_data, 'grle_name');
        }

        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        } else {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '职级管理', '启用', '启用' . implode(',', $find_name) . '成功');
            return $this->success('启用成功');
        }
    }

    // 停用
    public function stop(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->startValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();
        DB::beginTransaction();
        if (in_array($request['type'], [1])) {
            // 职称管理停用
            $return_data = GradeProfessionalService::stop($user_info, $request->all());
            // 获取停用的名称
            $field_columns = ['grpr_name'];
            $find_data = GradeProfessionalService::getFieldsByIdAll($user_info, $request->all(), $field_columns);
            $find_name = array_column($find_data, 'grpr_name');
        } else {
            // 级别管理
            $return_data = GradeLevelService::stop($user_info, $request->all());
            // 获取停用的名称
            $field_columns = ['grle_name'];
            $find_data = GradeLevelService::getFieldsByIdAll($user_info, $request->all(), $field_columns);
            $find_name = array_column($find_data, 'grle_name');
        }

        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        } else {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '职级管理', '停用', '停用' . implode(',', $find_name) . '成功');
            return $this->success('停用成功');
        }
    }

    // 删除
    public function delete(Request $request)
    {
        // 验证参数
        $error_message = $this->dataService->mainValidator->startValidator($request->all());
        if (!empty($error_message)) {
            return $this->error('请求参数错误！', ErrorInc::INVALID_PARAM, $error_message);
        }

        // 获取用户信息
        $user_info = LoginUserService::getUser();
        DB::beginTransaction();
        if (in_array($request['type'], [1])) {
            // 职称管理删除
            $return_data = GradeProfessionalService::remove($user_info, $request->all());
            // 获取删除的名称
            $field_columns = ['grpr_name'];
            $find_data = GradeProfessionalService::getFieldsByIdAll($user_info, $request->all(), $field_columns);
            $find_name = array_column($find_data, 'grpr_name');
        } else {
            // 级别管理
            $return_data = GradeLevelService::remove($user_info, $request->all());
            // 获取删除的名称
            $field_columns = ['grle_name'];
            $find_data = GradeLevelService::getFieldsByIdAll($user_info, $request->all(), $field_columns);
            $find_name = array_column($find_data, 'grle_name');
        }

        if ($return_data['code'] != 200) {
            DB::rollBack();
            return response()->json($return_data);
        } else {
            DB::commit();
            OperationLog::addLog($user_info, getRealIp($request), '职级管理', '删除', '删除' . implode(',', $find_name) . '成功');
            return $this->success('删除成功');
        }
    }

    // 获取级别
    public function getGradeLevel(Request $request)
    {
        // 获取用户信息
        $user_info = LoginUserService::getUser();

        // 获取级别信息
        $return_data = GradeLevelService::getGradeLevel($user_info, $request->all(), 10);
        return $this->success('成功', $return_data);
    }
}
